import { mysqlTable, serial, varchar, text, boolean, timestamp, int, decimal } from "drizzle-orm/mysql-core";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";

// Users table
export const users = mysqlTable("users", {
  id: serial("id").primaryKey(),
  username: varchar("username", { length: 255 }).notNull().unique(),
  password: varchar("password", { length: 255 }).notNull(),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertUserSchema = createInsertSchema(users).pick({
  username: true,
  password: true,
});

export type InsertUser = z.infer<typeof insertUserSchema>;
export type User = typeof users.$inferSelect;

// Categories table
export const categories = mysqlTable("categories", {
  id: serial("id").primaryKey(),
  name: varchar("name", { length: 255 }).notNull(),
  slug: varchar("slug", { length: 255 }).notNull().unique(),
  description: text("description"),
  isActive: boolean("is_active").default(true),
  sortOrder: int("sort_order").default(0),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertCategorySchema = createInsertSchema(categories).pick({
  name: true,
  slug: true,
  description: true,
  isActive: true,
  sortOrder: true,
});

export type InsertCategory = z.infer<typeof insertCategorySchema>;
export type Category = typeof categories.$inferSelect;

// Menu items table
export const menuItems = mysqlTable("menu_items", {
  id: serial("id").primaryKey(),
  name: varchar("name", { length: 255 }).notNull(),
  description: text("description"),
  price: varchar("price", { length: 20 }).notNull(),
  categoryId: int("category_id").notNull(),
  image: text("image"),
  images: text("images"),
  isActive: boolean("is_active").default(true),
  isVegetarian: boolean("is_vegetarian").default(false),
  isVegan: boolean("is_vegan").default(false),
  isGlutenFree: boolean("is_gluten_free").default(false),
  isSpicy: boolean("is_spicy").default(false),
  isChefSpecial: boolean("is_chef_special").default(false),
  isMonthlySpecial: boolean("is_monthly_special").default(false),
  isCampaign: boolean("is_campaign").default(false),
  campaignPrice: varchar("campaign_price", { length: 20 }),
  sortOrder: int("sort_order").default(0),
  preparationTime: varchar("preparation_time", { length: 50 }),
  allergens: text("allergens"),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertMenuItemSchema = createInsertSchema(menuItems).pick({
  name: true,
  description: true,
  price: true,
  categoryId: true,
  image: true,
  images: true,
  isActive: true,
  isVegetarian: true,
  isVegan: true,
  isGlutenFree: true,
  isSpicy: true,
  isChefSpecial: true,
  isMonthlySpecial: true,
  isCampaign: true,
  campaignPrice: true,
  sortOrder: true,
  preparationTime: true,
  allergens: true,
});

export type InsertMenuItem = z.infer<typeof insertMenuItemSchema>;
export type MenuItem = typeof menuItems.$inferSelect;

// Restaurant information
export const restaurants = mysqlTable("restaurants", {
  id: serial("id").primaryKey(),
  name: varchar("name", { length: 255 }).notNull(),
  description: text("description"),
  address: text("address"),
  phone: varchar("phone", { length: 20 }),
  email: varchar("email", { length: 255 }),
  website: varchar("website", { length: 255 }),
  logo: text("logo"),
  isActive: boolean("is_active").default(true),
  // Payment system settings
  enablePayments: boolean("enable_payments").default(false),
  enablePaytr: boolean("enable_paytr").default(false),
  enableIyzico: boolean("enable_iyzico").default(false),
  enableCashPayment: boolean("enable_cash_payment").default(true),
  enableOrderSystem: boolean("enable_order_system").default(false),
  // Commission rates
  paytrCommissionRate: varchar("paytr_commission_rate", { length: 10 }).default("2.9"), // %2.9
  iyzicoCommissionRate: varchar("iyzico_commission_rate", { length: 10 }).default("2.7"), // %2.7
  fixedFeePaytr: varchar("fixed_fee_paytr", { length: 10 }).default("0.25"), // 0.25 TL
  fixedFeeIyzico: varchar("fixed_fee_iyzico", { length: 10 }).default("0.30"), // 0.30 TL
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertRestaurantSchema = createInsertSchema(restaurants).pick({
  name: true,
  description: true,
  address: true,
  phone: true,
  email: true,
  website: true,
  logo: true,
  isActive: true,
  enablePayments: true,
  enablePaytr: true,
  enableIyzico: true,
  enableCashPayment: true,
  enableOrderSystem: true,
  paytrCommissionRate: true,
  iyzicoCommissionRate: true,
  fixedFeePaytr: true,
  fixedFeeIyzico: true,
});

export type InsertRestaurant = z.infer<typeof insertRestaurantSchema>;
export type Restaurant = typeof restaurants.$inferSelect;

// Theme settings
export const themeSettings = mysqlTable("theme_settings", {
  id: serial("id").primaryKey(),
  name: varchar("name", { length: 255 }).notNull(),
  primaryColor: varchar("primary_color", { length: 7 }).notNull(),
  secondaryColor: varchar("secondary_color", { length: 7 }).notNull(),
  backgroundColor: varchar("background_color", { length: 7 }).notNull(),
  textColor: varchar("text_color", { length: 7 }).notNull(),
  fontFamily: varchar("font_family", { length: 100 }).notNull(),
  isActive: boolean("is_active").default(false),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertThemeSettingsSchema = createInsertSchema(themeSettings).pick({
  name: true,
  primaryColor: true,
  secondaryColor: true,
  backgroundColor: true,
  textColor: true,
  fontFamily: true,
  isActive: true,
});

export type InsertThemeSettings = z.infer<typeof insertThemeSettingsSchema>;
export type ThemeSettings = typeof themeSettings.$inferSelect;

// Images table
export const images = mysqlTable("images", {
  id: serial("id").primaryKey(),
  filename: varchar("filename", { length: 255 }).notNull(),
  originalName: varchar("original_name", { length: 255 }).notNull(),
  mimeType: varchar("mime_type", { length: 100 }).notNull(),
  size: int("size").notNull(),
  url: text("url").notNull(),
  alt: text("alt"),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertImageSchema = createInsertSchema(images).pick({
  filename: true,
  originalName: true,
  mimeType: true,
  size: true,
  url: true,
  alt: true,
});

export type InsertImage = z.infer<typeof insertImageSchema>;
export type Image = typeof images.$inferSelect;

// Menu item images junction table
export const menuItemImages = mysqlTable("menu_item_images", {
  id: serial("id").primaryKey(),
  menuItemId: int("menu_item_id").notNull(),
  imageId: int("image_id").notNull(),
  order: int("order").default(0),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertMenuItemImageSchema = createInsertSchema(menuItemImages).pick({
  menuItemId: true,
  imageId: true,
  order: true,
});

export type InsertMenuItemImage = z.infer<typeof insertMenuItemImageSchema>;
export type MenuItemImage = typeof menuItemImages.$inferSelect;

// SEO settings
export const seoSettings = mysqlTable("seo_settings", {
  id: serial("id").primaryKey(),
  pageTitle: varchar("page_title", { length: 255 }),
  metaDescription: text("meta_description"),
  metaKeywords: text("meta_keywords"),
  ogTitle: varchar("og_title", { length: 255 }),
  ogDescription: text("og_description"),
  ogImage: text("og_image"),
  twitterTitle: varchar("twitter_title", { length: 255 }),
  twitterDescription: text("twitter_description"),
  twitterImage: text("twitter_image"),
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertSeoSettingsSchema = createInsertSchema(seoSettings).pick({
  pageTitle: true,
  metaDescription: true,
  metaKeywords: true,
  ogTitle: true,
  ogDescription: true,
  ogImage: true,
  twitterTitle: true,
  twitterDescription: true,
  twitterImage: true,
  isActive: true,
});

export type InsertSeoSettings = z.infer<typeof insertSeoSettingsSchema>;
export type SeoSettings = typeof seoSettings.$inferSelect;

// E-commerce tables

// Orders table
export const orders = mysqlTable("orders", {
  id: serial("id").primaryKey(),
  customerId: int("customer_id"),
  orderNumber: varchar("order_number", { length: 50 }).notNull().unique(),
  status: varchar("status", { length: 50 }).notNull().default("pending"),
  totalAmount: varchar("total_amount", { length: 20 }).notNull(),
  currency: varchar("currency", { length: 3 }).default("TRY"),
  paymentMethod: varchar("payment_method", { length: 50 }),
  paymentStatus: varchar("payment_status", { length: 50 }).default("pending"),
  notes: text("notes"),
  customerName: varchar("customer_name", { length: 255 }),
  customerPhone: varchar("customer_phone", { length: 20 }),
  customerEmail: varchar("customer_email", { length: 255 }),
  deliveryAddress: text("delivery_address"),
  deliveryDate: timestamp("delivery_date"),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertOrderSchema = createInsertSchema(orders).pick({
  customerId: true,
  orderNumber: true,
  status: true,
  totalAmount: true,
  currency: true,
  paymentMethod: true,
  paymentStatus: true,
  notes: true,
  customerName: true,
  customerPhone: true,
  customerEmail: true,
  deliveryAddress: true,
  deliveryDate: true,
});

export type InsertOrder = z.infer<typeof insertOrderSchema>;
export type Order = typeof orders.$inferSelect;

// Order items table
export const orderItems = mysqlTable("order_items", {
  id: serial("id").primaryKey(),
  orderId: int("order_id").notNull(),
  menuItemId: int("menu_item_id").notNull(),
  quantity: int("quantity").notNull(),
  price: varchar("price", { length: 20 }).notNull(),
  notes: text("notes"),
});

export const insertOrderItemSchema = createInsertSchema(orderItems).pick({
  orderId: true,
  menuItemId: true,
  quantity: true,
  price: true,
  notes: true,
});

export type InsertOrderItem = z.infer<typeof insertOrderItemSchema>;
export type OrderItem = typeof orderItems.$inferSelect;

// Customers table
export const customers = mysqlTable("customers", {
  id: serial("id").primaryKey(),
  name: varchar("name", { length: 255 }).notNull(),
  phone: varchar("phone", { length: 20 }),
  email: varchar("email", { length: 255 }),
  address: text("address"),
  points: int("points").default(0),
  totalSpent: varchar("total_spent", { length: 20 }).default("0"),
  totalOrders: int("total_orders").default(0),
  visits: int("visits").default(0),
  lastVisit: timestamp("last_visit"),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertCustomerSchema = createInsertSchema(customers).pick({
  name: true,
  phone: true,
  email: true,
  address: true,
  points: true,
  totalSpent: true,
  totalOrders: true,
  visits: true,
  lastVisit: true,
});

export type InsertCustomer = z.infer<typeof insertCustomerSchema>;
export type Customer = typeof customers.$inferSelect;

// Payments table
export const payments = mysqlTable("payments", {
  id: serial("id").primaryKey(),
  orderId: int("order_id").notNull(),
  amount: varchar("amount", { length: 20 }).notNull(),
  currency: varchar("currency", { length: 3 }).default("TRY"),
  method: varchar("method", { length: 50 }).notNull(),
  status: varchar("status", { length: 50 }).notNull(),
  transactionId: varchar("transaction_id", { length: 255 }),
  providerId: varchar("provider_id", { length: 255 }),
  providerResponse: text("provider_response"),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertPaymentSchema = createInsertSchema(payments).pick({
  orderId: true,
  amount: true,
  currency: true,
  method: true,
  status: true,
  transactionId: true,
  providerId: true,
  providerResponse: true,
});

export type InsertPayment = z.infer<typeof insertPaymentSchema>;
export type Payment = typeof payments.$inferSelect;

// Coupons table
export const coupons = mysqlTable("coupons", {
  id: serial("id").primaryKey(),
  code: varchar("code", { length: 50 }).notNull().unique(),
  name: varchar("name", { length: 255 }).notNull(),
  description: text("description"),
  discountType: varchar("discount_type", { length: 20 }).notNull(), // percentage, fixed
  discountValue: varchar("discount_value", { length: 20 }).notNull(),
  minOrderAmount: varchar("min_order_amount", { length: 20 }).default("0"),
  maxUses: int("max_uses").default(1),
  usedCount: int("used_count").default(0),
  isActive: boolean("is_active").default(true),
  expiresAt: timestamp("expires_at"),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertCouponSchema = createInsertSchema(coupons).pick({
  code: true,
  name: true,
  description: true,
  discountType: true,
  discountValue: true,
  minOrderAmount: true,
  maxUses: true,
  usedCount: true,
  isActive: true,
  expiresAt: true,
});

export type InsertCoupon = z.infer<typeof insertCouponSchema>;
export type Coupon = typeof coupons.$inferSelect;

// Analytics table
export const analytics = mysqlTable("analytics", {
  id: serial("id").primaryKey(),
  event: varchar("event", { length: 100 }).notNull(),
  data: text("data"),
  userId: int("user_id"),
  sessionId: varchar("session_id", { length: 255 }),
  ipAddress: varchar("ip_address", { length: 45 }),
  userAgent: text("user_agent"),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertAnalyticsSchema = createInsertSchema(analytics).pick({
  event: true,
  data: true,
  userId: true,
  sessionId: true,
  ipAddress: true,
  userAgent: true,
});

export type InsertAnalytics = z.infer<typeof insertAnalyticsSchema>;
export type Analytics = typeof analytics.$inferSelect;

// Notifications table
export const notifications = mysqlTable("notifications", {
  id: serial("id").primaryKey(),
  title: varchar("title", { length: 255 }).notNull(),
  message: text("message").notNull(),
  type: varchar("type", { length: 50 }).notNull(), // info, success, warning, error
  isRead: boolean("is_read").default(false),
  userId: int("user_id"),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertNotificationSchema = createInsertSchema(notifications).pick({
  title: true,
  message: true,
  type: true,
  isRead: true,
  userId: true,
});

export type InsertNotification = z.infer<typeof insertNotificationSchema>;
export type Notification = typeof notifications.$inferSelect;

// Reviews table
export const reviews = mysqlTable("reviews", {
  id: serial("id").primaryKey(),
  menuItemId: int("menu_item_id"),
  customerName: varchar("customer_name", { length: 255 }).notNull(),
  customerEmail: varchar("customer_email", { length: 255 }),
  rating: int("rating").notNull(), // 1-5
  comment: text("comment"),
  isApproved: boolean("is_approved").default(false),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertReviewSchema = createInsertSchema(reviews).pick({
  menuItemId: true,
  customerName: true,
  customerEmail: true,
  rating: true,
  comment: true,
  isApproved: true,
});

export type InsertReview = z.infer<typeof insertReviewSchema>;
export type Review = typeof reviews.$inferSelect;

// Payment webhooks table for tracking payment provider callbacks
export const paymentWebhooks = mysqlTable("payment_webhooks", {
  id: serial("id").primaryKey(),
  provider: varchar("provider", { length: 50 }).notNull(), // paytr, iyzico
  webhookData: text("webhook_data").notNull(),
  orderId: int("order_id"),
  status: varchar("status", { length: 50 }).notNull(),
  processed: boolean("processed").default(false),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertPaymentWebhookSchema = createInsertSchema(paymentWebhooks).pick({
  provider: true,
  webhookData: true,
  orderId: true,
  status: true,
  processed: true,
});

export type InsertPaymentWebhook = z.infer<typeof insertPaymentWebhookSchema>;
export type PaymentWebhook = typeof paymentWebhooks.$inferSelect;