import { 
  User, 
  InsertUser, 
  Category, 
  InsertCategory, 
  MenuItem, 
  InsertMenuItem, 
  Restaurant, 
  InsertRestaurant,
  ThemeSettings,
  InsertThemeSettings,
  Image,
  InsertImage,
  MenuItemImage,
  SeoSettings,
  InsertSeoSettings,
  Order,
  InsertOrder,
  OrderItem,
  InsertOrderItem,
  Customer,
  InsertCustomer,
  Coupon,
  InsertCoupon,
  Analytics,
  InsertAnalytics,
  Notification,
  InsertNotification,
  Review,
  InsertReview
} from "@shared/schema";
import session from "express-session";
import createMemoryStore from "memorystore";

const MemoryStore = createMemoryStore(session);

export interface IStorage {
  // User methods
  getUser(id: number): Promise<User | undefined>;
  getUserByUsername(username: string): Promise<User | undefined>;
  createUser(insertUser: InsertUser): Promise<User>;

  // Category methods
  getCategories(): Promise<Category[]>;
  getCategoryBySlug(slug: string): Promise<Category | undefined>;
  createCategory(category: InsertCategory): Promise<Category>;
  updateCategory(id: number, category: Partial<InsertCategory>): Promise<Category>;
  deleteCategory(id: number): Promise<boolean>;

  // Menu Item methods
  getMenuItems(): Promise<MenuItem[]>;
  getMenuItem(id: number): Promise<MenuItem | undefined>;
  getMenuItemsByCategory(categoryId: number): Promise<MenuItem[]>;
  createMenuItem(menuItem: InsertMenuItem): Promise<MenuItem>;
  updateMenuItem(id: number, menuItem: Partial<InsertMenuItem>): Promise<MenuItem>;
  deleteMenuItem(id: number): Promise<boolean>;

  // Restaurant methods
  getRestaurantInfo(): Promise<Restaurant | undefined>;
  createRestaurantInfo(info: InsertRestaurant): Promise<Restaurant>;
  updateRestaurantInfo(id: number, info: Partial<InsertRestaurant>): Promise<Restaurant>;

  // Theme methods
  getThemeSettings(): Promise<ThemeSettings | undefined>;
  getActiveTheme(): Promise<ThemeSettings | undefined>;
  createThemeSettings(settings: InsertThemeSettings): Promise<ThemeSettings>;
  updateThemeSettings(id: number, settings: Partial<InsertThemeSettings>): Promise<ThemeSettings>;
  activateTheme(id: number): Promise<ThemeSettings>;
  deleteThemeSettings(id: number): Promise<boolean>;

  // Image methods
  getImages(): Promise<Image[]>;
  getImage(id: number): Promise<Image | undefined>;
  createImage(image: InsertImage): Promise<Image>;
  updateImage(id: number, image: Partial<InsertImage>): Promise<Image>;
  deleteImage(id: number): Promise<boolean>;

  // Menu Item Image methods
  getMenuItemImages(menuItemId: number): Promise<Image[]>;
  addImageToMenuItem(menuItemId: number, imageId: number, order?: number): Promise<MenuItemImage>;
  removeImageFromMenuItem(menuItemId: number, imageId: number): Promise<boolean>;

  // SEO methods
  getSeoSettings(): Promise<SeoSettings | undefined>;
  createSeoSettings(settings: InsertSeoSettings): Promise<SeoSettings>;
  updateSeoSettings(id: number, settings: Partial<InsertSeoSettings>): Promise<SeoSettings>;
  deleteSeoSettings(id: number): Promise<boolean>;

  // Order methods
  getOrders(): Promise<Order[]>;
  getOrder(id: number): Promise<Order | undefined>;
  getOrdersByStatus(status: string): Promise<Order[]>;
  createOrder(order: InsertOrder): Promise<Order>;
  updateOrderStatus(id: number, status: string): Promise<Order>;
  deleteOrder(id: number): Promise<boolean>;

  // Order Item methods
  getOrderItems(orderId: number): Promise<OrderItem[]>;
  createOrderItem(orderItem: InsertOrderItem): Promise<OrderItem>;
  updateOrderItem(id: number, orderItem: Partial<InsertOrderItem>): Promise<OrderItem>;
  deleteOrderItem(id: number): Promise<boolean>;

  // Customer methods
  getCustomers(): Promise<Customer[]>;
  getCustomer(id: number): Promise<Customer | undefined>;
  getCustomerByPhone(phone: string): Promise<Customer | undefined>;
  createCustomer(customer: InsertCustomer): Promise<Customer>;
  updateCustomer(id: number, customer: Partial<InsertCustomer>): Promise<Customer>;
  addCustomerPoints(id: number, points: number): Promise<Customer>;

  // Coupon methods
  getCoupons(): Promise<Coupon[]>;
  getCoupon(id: number): Promise<Coupon | undefined>;
  getCouponByCode(code: string): Promise<Coupon | undefined>;
  createCoupon(coupon: InsertCoupon): Promise<Coupon>;
  updateCoupon(id: number, coupon: Partial<InsertCoupon>): Promise<Coupon>;
  useCoupon(id: number): Promise<Coupon>;
  deleteCoupon(id: number): Promise<boolean>;

  // Analytics methods
  getAnalytics(limit?: number): Promise<Analytics[]>;
  createAnalyticsEvent(analytics: InsertAnalytics): Promise<Analytics>;
  getAnalyticsByEvent(event: string, limit?: number): Promise<Analytics[]>;

  // Notification methods
  getNotifications(): Promise<Notification[]>;
  getUnreadNotifications(): Promise<Notification[]>;
  createNotification(notification: InsertNotification): Promise<Notification>;
  markNotificationAsRead(id: number): Promise<Notification>;
  deleteNotification(id: number): Promise<boolean>;

  // Review methods
  getReviews(): Promise<Review[]>;
  getApprovedReviews(): Promise<Review[]>;
  getMenuItemReviews(menuItemId: number): Promise<Review[]>;
  createReview(review: InsertReview): Promise<Review>;
  updateReview(id: number, review: Partial<InsertReview>): Promise<Review>;
  approveReview(id: number): Promise<Review>;
  deleteReview(id: number): Promise<boolean>;

  // Customer methods
  getCustomers(): Promise<Customer[]>;
  getCustomer(id: number): Promise<Customer | undefined>;
  getCustomerByPhone(phone: string): Promise<Customer | undefined>;
  createCustomer(customer: InsertCustomer): Promise<Customer>;
  updateCustomer(id: number, customer: Partial<InsertCustomer>): Promise<Customer>;
  deleteCustomer(id: number): Promise<boolean>;

  // Coupon methods
  getCoupons(): Promise<Coupon[]>;
  getCoupon(id: number): Promise<Coupon | undefined>;
  getCouponByCode(code: string): Promise<Coupon | undefined>;
  createCoupon(coupon: InsertCoupon): Promise<Coupon>;
  updateCoupon(id: number, coupon: Partial<InsertCoupon>): Promise<Coupon>;
  deleteCoupon(id: number): Promise<boolean>;

  // Session store
  sessionStore: any;
}

export class MemStorage implements IStorage {
  private usersMap = new Map<number, User>();
  private categoriesMap = new Map<number, Category>();
  private menuItemsMap = new Map<number, MenuItem>();
  private restaurantMap = new Map<number, Restaurant>();
  private themeSettingsMap = new Map<number, ThemeSettings>();
  private imagesMap = new Map<number, Image>();
  private menuItemImagesMap = new Map<string, MenuItemImage>();
  private seoSettingsMap = new Map<number, SeoSettings>();
  private ordersMap = new Map<number, any>();
  private orderItemsMap = new Map<number, any>();
  private customersMap = new Map<number, any>();
  private couponsMap = new Map<number, any>();
  private analyticsMap = new Map<number, any>();
  private notificationsMap = new Map<number, any>();
  private reviewsMap = new Map<number, any>();

  private currentUserId = 1;
  private currentCategoryId = 1;
  private currentMenuItemId = 1;
  private currentRestaurantId = 1;
  private currentThemeId = 1;
  private currentImageId = 1;
  private currentSeoId = 1;
  private currentOrderId = 1;
  private currentOrderItemId = 1;
  private currentCustomerId = 1;
  private currentCouponId = 1;
  private currentAnalyticsId = 1;
  private currentNotificationId = 1;
  private currentReviewId = 1;

  public sessionStore: any;

  constructor() {
    this.sessionStore = new MemoryStore({
      checkPeriod: 86400000,
    });
    this.seedData();
  }

  private seedData() {
    // Create admin user
    const adminUser: User = {
      id: 1,
      username: "admin",
      password: "$2b$10$rQ8K8vKjzGvKjzGvKjzGv.V8KjzGvKjzGvKjzGvKjzGvKjzGvKjzG",
      createdAt: new Date().toISOString()
    };
    this.usersMap.set(1, adminUser);
    this.currentUserId = 2;

    // Create categories
    const kahvaltilar: Category = {
      id: 1,
      name: "Kahvaltılar",
      slug: "kahvaltilar",
      description: "Taze ve lezzetli kahvaltı seçenekleri",
      image: null,
      isActive: true,
      createdAt: new Date().toISOString()
    };

    const anaYemekler: Category = {
      id: 2,
      name: "Ana Yemekler",
      slug: "ana-yemekler",
      description: "Doyurucu ana yemek seçenekleri",
      image: null,
      isActive: true,
      createdAt: new Date().toISOString()
    };

    this.categoriesMap.set(1, kahvaltilar);
    this.categoriesMap.set(2, anaYemekler);
    this.currentCategoryId = 3;

    // Create menu items with SVG images
    const items = [
      {
        id: 1,
        name: "Menemen",
        description: "Domates, biber ve soğan ile hazırlanmış geleneksel lezzet",
        price: "25.00",
        categoryId: 1,
        image: "data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNTAwIiBoZWlnaHQ9IjMwMCIgdmlld0JveD0iMCAwIDUwMCAzMDAiIGZpbGw9Im5vbmUiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+CjxyZWN0IHdpZHRoPSI1MDAiIGhlaWdodD0iMzAwIiBmaWxsPSIjZjk3MzE2Ii8+Cjx0ZXh0IHg9IjI1MCIgeT0iMTUwIiBmb250LWZhbWlseT0iQXJpYWwiIGZvbnQtc2l6ZT0iMjQiIGZvbnQtd2VpZ2h0PSJib2xkIiBmaWxsPSJ3aGl0ZSIgdGV4dC1hbmNob3I9Im1pZGRsZSIgZG9taW5hbnQtYmFzZWxpbmU9Im1pZGRsZSI+TWVuZW1lbjwvdGV4dD4KPC9zdmc+",
        images: null,
        isActive: true,
        isVegetarian: true,
        isVegan: false,
        isGlutenFree: false,
        isSpicy: false,
        preparationTime: "10",
        allergens: "Yumurta",
        createdAt: new Date().toISOString()
      },
      {
        id: 2,
        name: "Sucuklu Yumurta",
        description: "Taze yumurta ve kaliteli sucuk",
        price: "30.00",
        categoryId: 1,
        image: "data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNTAwIiBoZWlnaHQ9IjMwMCIgdmlld0JveD0iMCAwIDUwMCAzMDAiIGZpbGw9Im5vbmUiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+CjxyZWN0IHdpZHRoPSI1MDAiIGhlaWdodD0iMzAwIiBmaWxsPSIjZWY0NDQ0Ii8+Cjx0ZXh0IHg9IjI1MCIgeT0iMTUwIiBmb250LWZhbWlseT0iQXJpYWwiIGZvbnQtc2l6ZT0iMjAiIGZvbnQtd2VpZ2h0PSJib2xkIiBmaWxsPSJ3aGl0ZSIgdGV4dC1hbmNob3I9Im1pZGRsZSIgZG9taW5hbnQtYmFzZWxpbmU9Im1pZGRsZSI+U3VjdWtsdSBZdW11cnRhPC90ZXh0Pgo8L3N2Zz4=",
        images: null,
        isActive: true,
        isVegetarian: false,
        isVegan: false,
        isGlutenFree: true,
        isSpicy: false,
        preparationTime: "12",
        allergens: "Yumurta",
        createdAt: new Date().toISOString()
      },
      {
        id: 3,
        name: "Fatih",
        description: "asda",
        price: "20.00",
        categoryId: 1,
        image: "data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNTAwIiBoZWlnaHQ9IjMwMCIgdmlld0JveD0iMCAwIDUwMCAzMDAiIGZpbGw9Im5vbmUiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+CjxyZWN0IHdpZHRoPSI1MDAiIGhlaWdodD0iMzAwIiBmaWxsPSIjMTBiOTgxIi8+Cjx0ZXh0IHg9IjI1MCIgeT0iMTUwIiBmb250LWZhbWlseT0iQXJpYWwiIGZvbnQtc2l6ZT0iMjQiIGZvbnQtd2VpZ2h0PSJib2xkIiBmaWxsPSJ3aGl0ZSIgdGV4dC1hbmNob3I9Im1pZGRsZSIgZG9taW5hbnQtYmFzZWxpbmU9Im1pZGRsZSI+RmF0aWg8L3RleHQ+Cjwvc3ZnPg==",
        images: null,
        isActive: true,
        isVegetarian: false,
        isVegan: false,
        isGlutenFree: false,
        isSpicy: false,
        preparationTime: "10",
        allergens: null,
        createdAt: new Date().toISOString()
      },
      {
        id: 4,
        name: "Köfte Izgara",
        description: "Ev yapımı köfte ızgara",
        price: "45.00",
        categoryId: 2,
        image: "data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNTAwIiBoZWlnaHQ9IjMwMCIgdmlld0JveD0iMCAwIDUwMCAzMDAiIGZpbGw9Im5vbmUiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+CjxyZWN0IHdpZHRoPSI1MDAiIGhlaWdodD0iMzAwIiBmaWxsPSIjOWZhOGRhIi8+Cjx0ZXh0IHg9IjI1MCIgeT0iMTUwIiBmb250LWZhbWlseT0iQXJpYWwiIGZvbnQtc2l6ZT0iMjQiIGZvbnQtd2VpZ2h0PSJib2xkIiBmaWxsPSJ3aGl0ZSIgdGV4dC1hbmNob3I9Im1pZGRsZSIgZG9taW5hbnQtYmFzZWxpbmU9Im1pZGRsZSI+S8O2ZnRlIEl6Z2FyYTwvdGV4dD4KPC9zdmc+",
        images: null,
        isActive: true,
        isVegetarian: false,
        isVegan: false,
        isGlutenFree: false,
        isSpicy: false,
        preparationTime: "20",
        allergens: null,
        createdAt: new Date().toISOString()
      },
      {
        id: 5,
        name: "Balık Izgara",
        description: "Taze balık ızgara",
        price: "50.00",
        categoryId: 2,
        image: "data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNTAwIiBoZWlnaHQ9IjMwMCIgdmlld0JveD0iMCAwIDUwMCAzMDAiIGZpbGw9Im5vbmUiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+CjxyZWN0IHdpZHRoPSI1MDAiIGhlaWdodD0iMzAwIiBmaWxsPSIjMDZiNmQ0Ci8+Cjx0ZXh0IHg9IjI1MCIgeT0iMTUwIiBmb250LWZhbWlseT0iQXJpYWwiIGZvbnQtc2l6ZT0iMjQiIGZvbnQtd2VpZ2h0PSJib2xkIiBmaWxsPSJ3aGl0ZSIgdGV4dC1hbmNob3I9Im1pZGRsZSIgZG9taW5hbnQtYmFzZWxpbmU9Im1pZGRsZSI+QmFsxLFrIEl6Z2FyYTwvdGV4dD4KPC9zdmc+",
        images: null,
        isActive: true,
        isVegetarian: false,
        isVegan: false,
        isGlutenFree: false,
        isSpicy: false,
        preparationTime: "25",
        allergens: "Balık",
        createdAt: new Date().toISOString()
      }
    ];

    items.forEach(item => this.menuItemsMap.set(item.id, item));
    this.currentMenuItemId = 6;

    // Create restaurant info
    const restaurant: Restaurant = {
      id: 1,
      name: "Jouska Cafe & Restaurant",
      tagline: "Lezzet ve kalitenin buluşma noktası",
      description: "Modern atmosferde geleneksel lezzetler",
      address: "İstanbul, Türkiye",
      phone: "+90 212 XXX XX XX",
      email: "info@jouska.com",
      website: "https://jouska.com",
      logo: null,
      workingHours: "08:00 - 23:00",
      googleMyBusinessUrl: null,
      googlePlaceId: null,
      isActive: true,
      createdAt: new Date().toISOString()
    };
    this.restaurantMap.set(1, restaurant);

    // Create theme settings
    const theme: ThemeSettings = {
      id: 1,
      name: "Default Theme",
      primaryColor: "#2563eb",
      secondaryColor: "#64748b",
      accentColor: "#f59e0b",
      backgroundColor: "#ffffff",
      textColor: "#1f2937",
      fontFamily: "Inter",
      isActive: true,
      createdAt: new Date().toISOString()
    };
    this.themeSettingsMap.set(1, theme);

    // Create SEO settings
    const seoSettings: SeoSettings = {
      id: 1,
      pageTitle: "Jouska Cafe & Restaurant - Lezzetli Yemekler ve Kahve",
      metaDescription: "Jouska Cafe & Restaurant'ta taze kahveler, nefis yemekler ve sıcak atmosferin keyfini çıkarın. Kahvaltı, ana yemek ve tatlı seçenekleriyle sizleri bekliyoruz.",
      metaKeywords: "jouska, cafe, restaurant, kahve, yemek, kahvaltı, tatlı, istanbul",
      ogTitle: "Jouska Cafe & Restaurant - Eşsiz Lezzetler",
      ogDescription: "Jouska'da her lokmada kalite ve lezzeti bir arada yaşayın. Özel menümüz ve samimi atmosferimizle sizleri ağırlıyoruz.",
      ogImage: "/images/jouska-og-image.jpg",
      twitterTitle: "Jouska Cafe & Restaurant",
      twitterDescription: "Taze kahveler ve nefis yemeklerle Jouska deneyimi.",
      twitterImage: "/images/jouska-twitter-image.jpg",
      canonicalUrl: "https://jouskamenu.com",
      robotsContent: "index, follow",
      structuredData: JSON.stringify({
        "@context": "https://schema.org",
        "@type": "Restaurant",
        "name": "Jouska Cafe & Restaurant",
        "description": "Modern cafe ve restaurant deneyimi",
        "url": "https://jouskamenu.com",
        "telephone": "+90 XXX XXX XX XX",
        "address": {
          "@type": "PostalAddress",
          "addressCountry": "TR"
        },
        "servesCuisine": ["Turkish", "International"],
        "priceRange": "$$"
      }),
      isActive: true,
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString()
    };
    this.seoSettingsMap.set(1, seoSettings);
  }

  // User methods
  async getUser(id: number): Promise<User | undefined> {
    return this.usersMap.get(id);
  }

  async getUserByUsername(username: string): Promise<User | undefined> {
    return Array.from(this.usersMap.values()).find(user => user.username === username);
  }

  async createUser(insertUser: InsertUser): Promise<User> {
    const id = this.currentUserId++;
    const user: User = { 
      ...insertUser, 
      id,
      createdAt: new Date().toISOString()
    };
    this.usersMap.set(id, user);
    return user;
  }

  // Category methods
  async getCategories(): Promise<Category[]> {
    return Array.from(this.categoriesMap.values());
  }

  async getCategoryBySlug(slug: string): Promise<Category | undefined> {
    return Array.from(this.categoriesMap.values()).find(cat => cat.slug === slug);
  }

  async createCategory(insertCategory: InsertCategory): Promise<Category> {
    const id = this.currentCategoryId++;
    const category: Category = { 
      ...insertCategory, 
      id,
      createdAt: new Date().toISOString()
    };
    this.categoriesMap.set(id, category);
    return category;
  }

  async updateCategory(id: number, updateData: Partial<InsertCategory>): Promise<Category> {
    const existingCategory = this.categoriesMap.get(id);
    if (!existingCategory) {
      throw new Error(`Category with id ${id} not found`);
    }

    const updatedCategory: Category = {
      ...existingCategory,
      ...updateData
    };

    this.categoriesMap.set(id, updatedCategory);
    return updatedCategory;
  }

  async deleteCategory(id: number): Promise<boolean> {
    return this.categoriesMap.delete(id);
  }

  // Menu Item methods
  async getMenuItems(): Promise<MenuItem[]> {
    return Array.from(this.menuItemsMap.values());
  }

  async getMenuItem(id: number): Promise<MenuItem | undefined> {
    return this.menuItemsMap.get(id);
  }

  async getMenuItemsByCategory(categoryId: number): Promise<MenuItem[]> {
    return Array.from(this.menuItemsMap.values())
      .filter(item => item.categoryId === categoryId);
  }

  async createMenuItem(insertMenuItem: InsertMenuItem): Promise<MenuItem> {
    const id = this.currentMenuItemId++;
    const menuItem: MenuItem = { 
      ...insertMenuItem,
      id,
      isActive: insertMenuItem.isActive ?? true,
      isVegetarian: insertMenuItem.isVegetarian ?? false,
      isVegan: insertMenuItem.isVegan ?? false,
      isGlutenFree: insertMenuItem.isGlutenFree ?? false,
      isSpicy: insertMenuItem.isSpicy ?? false,
      isCampaign: insertMenuItem.isCampaign ?? false,
      sortOrder: insertMenuItem.sortOrder ?? 0,
      createdAt: new Date().toISOString()
    };
    
    this.menuItemsMap.set(id, menuItem);
    return menuItem;
  }

  async updateMenuItem(id: number, updateData: Partial<InsertMenuItem>): Promise<MenuItem> {
    const existingItem = this.menuItemsMap.get(id);
    if (!existingItem) {
      throw new Error(`Menu item with id ${id} not found`);
    }

    const updatedItem: MenuItem = {
      ...existingItem,
      ...updateData
    };

    this.menuItemsMap.set(id, updatedItem);
    return updatedItem;
  }

  async deleteMenuItem(id: number): Promise<boolean> {
    return this.menuItemsMap.delete(id);
  }

  // Restaurant methods
  async getRestaurantInfo(): Promise<Restaurant | undefined> {
    return Array.from(this.restaurantMap.values())[0];
  }

  async createRestaurantInfo(insertRestaurant: InsertRestaurant): Promise<Restaurant> {
    const id = this.currentRestaurantId++;
    const restaurant: Restaurant = { 
      ...insertRestaurant, 
      id,
      createdAt: new Date().toISOString()
    };
    this.restaurantMap.set(id, restaurant);
    return restaurant;
  }

  async updateRestaurantInfo(id: number, updateData: Partial<InsertRestaurant>): Promise<Restaurant> {
    const existingRestaurant = this.restaurantMap.get(id);
    if (!existingRestaurant) {
      throw new Error(`Restaurant with id ${id} not found`);
    }

    const updatedRestaurant: Restaurant = {
      ...existingRestaurant,
      ...updateData
    };

    this.restaurantMap.set(id, updatedRestaurant);
    return updatedRestaurant;
  }

  // Theme methods
  async getThemeSettings(): Promise<ThemeSettings | undefined> {
    return Array.from(this.themeSettingsMap.values())[0];
  }

  async getActiveTheme(): Promise<ThemeSettings | undefined> {
    return Array.from(this.themeSettingsMap.values()).find(theme => theme.isActive);
  }

  async createThemeSettings(insertTheme: InsertThemeSettings): Promise<ThemeSettings> {
    const id = this.currentThemeId++;
    const theme: ThemeSettings = { 
      ...insertTheme, 
      id,
      createdAt: new Date().toISOString()
    };
    this.themeSettingsMap.set(id, theme);
    return theme;
  }

  async updateThemeSettings(id: number, updateData: Partial<InsertThemeSettings>): Promise<ThemeSettings> {
    const existingTheme = this.themeSettingsMap.get(id);
    if (!existingTheme) {
      throw new Error(`Theme with id ${id} not found`);
    }

    const updatedTheme: ThemeSettings = {
      ...existingTheme,
      ...updateData
    };

    this.themeSettingsMap.set(id, updatedTheme);
    return updatedTheme;
  }

  async activateTheme(id: number): Promise<ThemeSettings> {
    // Deactivate all themes
    for (const theme of this.themeSettingsMap.values()) {
      theme.isActive = false;
    }

    // Activate the specified theme
    const theme = this.themeSettingsMap.get(id);
    if (!theme) {
      throw new Error(`Theme with id ${id} not found`);
    }

    theme.isActive = true;
    this.themeSettingsMap.set(id, theme);
    return theme;
  }

  async deleteThemeSettings(id: number): Promise<boolean> {
    return this.themeSettingsMap.delete(id);
  }

  // Image methods
  async getImages(): Promise<Image[]> {
    return Array.from(this.imagesMap.values());
  }

  async getImage(id: number): Promise<Image | undefined> {
    return this.imagesMap.get(id);
  }

  async createImage(insertImage: InsertImage): Promise<Image> {
    const id = this.currentImageId++;
    const image: Image = { 
      ...insertImage, 
      id,
      createdAt: new Date().toISOString()
    };
    this.imagesMap.set(id, image);
    return image;
  }

  async updateImage(id: number, updateData: Partial<InsertImage>): Promise<Image> {
    const existingImage = this.imagesMap.get(id);
    if (!existingImage) {
      throw new Error(`Image with id ${id} not found`);
    }

    const updatedImage: Image = {
      ...existingImage,
      ...updateData
    };

    this.imagesMap.set(id, updatedImage);
    return updatedImage;
  }

  async deleteImage(id: number): Promise<boolean> {
    return this.imagesMap.delete(id);
  }

  // Menu Item Image methods
  async getMenuItemImages(menuItemId: number): Promise<Image[]> {
    const menuItemImages = Array.from(this.menuItemImagesMap.values())
      .filter(mii => mii.menuItemId === menuItemId)
      .sort((a, b) => (a.sortOrder || 0) - (b.sortOrder || 0));

    const imageIds = menuItemImages.map(mii => mii.imageId);
    return imageIds.map(id => this.imagesMap.get(id)).filter(Boolean) as Image[];
  }

  async addImageToMenuItem(menuItemId: number, imageId: number, order: number = 0): Promise<MenuItemImage> {
    const key = `${menuItemId}-${imageId}`;
    const menuItemImage: MenuItemImage = {
      id: Date.now(),
      menuItemId,
      imageId,
      sortOrder: order,
      createdAt: new Date().toISOString()
    };

    this.menuItemImagesMap.set(key, menuItemImage);
    return menuItemImage;
  }

  async removeImageFromMenuItem(menuItemId: number, imageId: number): Promise<boolean> {
    const key = `${menuItemId}-${imageId}`;
    return this.menuItemImagesMap.delete(key);
  }

  // SEO methods
  async getSeoSettings(): Promise<SeoSettings | undefined> {
    return Array.from(this.seoSettingsMap.values())[0];
  }

  async createSeoSettings(insertSeo: InsertSeoSettings): Promise<SeoSettings> {
    const id = this.currentSeoId++;
    const seo: SeoSettings = { 
      ...insertSeo, 
      id,
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString()
    };
    this.seoSettingsMap.set(id, seo);
    return seo;
  }

  async updateSeoSettings(id: number, updateData: Partial<InsertSeoSettings>): Promise<SeoSettings> {
    const existingSeo = this.seoSettingsMap.get(id);
    if (!existingSeo) {
      throw new Error(`SEO settings with id ${id} not found`);
    }

    const updatedSeo: SeoSettings = {
      ...existingSeo,
      ...updateData,
      updatedAt: new Date().toISOString()
    };

    this.seoSettingsMap.set(id, updatedSeo);
    return updatedSeo;
  }

  async deleteSeoSettings(id: number): Promise<boolean> {
    return this.seoSettingsMap.delete(id);
  }

  // Order methods
  async getOrders(): Promise<Order[]> {
    return Array.from(this.ordersMap.values());
  }

  async getOrder(id: number): Promise<Order | undefined> {
    return this.ordersMap.get(id);
  }

  async getOrdersByStatus(status: string): Promise<Order[]> {
    return Array.from(this.ordersMap.values()).filter(order => order.status === status);
  }

  async createOrder(insertOrder: InsertOrder): Promise<Order> {
    const id = this.currentOrderId++;
    const order: Order = { 
      ...insertOrder, 
      id,
      status: "pending",
      orderNumber: `ORD-${Date.now()}-${id}`,
      createdAt: new Date(),
      updatedAt: new Date()
    };
    this.ordersMap.set(id, order);
    return order;
  }

  async updateOrder(id: number, updateData: Partial<InsertOrder>): Promise<Order> {
    const existingOrder = this.ordersMap.get(id);
    if (!existingOrder) {
      throw new Error(`Order with id ${id} not found`);
    }
    const updatedOrder: Order = {
      ...existingOrder,
      ...updateData,
      updatedAt: new Date()
    };
    this.ordersMap.set(id, updatedOrder);
    return updatedOrder;
  }

  async updateOrderStatus(id: number, status: string): Promise<Order> {
    return this.updateOrder(id, { status });
  }

  async deleteOrder(id: number): Promise<boolean> {
    return this.ordersMap.delete(id);
  }

  // Order Item methods
  async getOrderItems(orderId: number): Promise<OrderItem[]> {
    return Array.from(this.orderItemsMap.values()).filter(item => item.orderId === orderId);
  }

  async createOrderItem(insertOrderItem: InsertOrderItem): Promise<OrderItem> {
    const id = this.currentOrderItemId++;
    const orderItem: OrderItem = { 
      ...insertOrderItem, 
      id,
      createdAt: new Date()
    };
    this.orderItemsMap.set(id, orderItem);
    return orderItem;
  }

  async updateOrderItem(id: number, updateData: Partial<InsertOrderItem>): Promise<OrderItem> {
    const existingItem = this.orderItemsMap.get(id);
    if (!existingItem) {
      throw new Error(`Order item with id ${id} not found`);
    }
    const updatedItem: OrderItem = {
      ...existingItem,
      ...updateData,
    };
    this.orderItemsMap.set(id, updatedItem);
    return updatedItem;
  }

  async deleteOrderItem(id: number): Promise<boolean> {
    return this.orderItemsMap.delete(id);
  }

  // Customer methods
  async getCustomers(): Promise<Customer[]> {
    return Array.from(this.customersMap.values());
  }

  async getCustomer(id: number): Promise<Customer | undefined> {
    return this.customersMap.get(id);
  }

  async getCustomerByPhone(phone: string): Promise<Customer | undefined> {
    return Array.from(this.customersMap.values()).find(customer => customer.phone === phone);
  }

  async createCustomer(insertCustomer: InsertCustomer): Promise<Customer> {
    const id = this.currentCustomerId++;
    const customer: Customer = { 
      ...insertCustomer, 
      id,
      points: 0,
      totalOrders: 0,
      totalSpent: "0",
      visits: 1,
      lastVisit: new Date(),
      createdAt: new Date()
    };
    this.customersMap.set(id, customer);
    return customer;
  }

  async updateCustomer(id: number, updateData: Partial<InsertCustomer>): Promise<Customer> {
    const existingCustomer = this.customersMap.get(id);
    if (!existingCustomer) {
      throw new Error(`Customer with id ${id} not found`);
    }
    const updatedCustomer: Customer = {
      ...existingCustomer,
      ...updateData,
    };
    this.customersMap.set(id, updatedCustomer);
    return updatedCustomer;
  }

  async deleteCustomer(id: number): Promise<boolean> {
    return this.customersMap.delete(id);
  }

  async addCustomerPoints(id: number, points: number): Promise<Customer> {
    const customer = this.customersMap.get(id);
    if (!customer) {
      throw new Error(`Customer with id ${id} not found`);
    }
    customer.points = (customer.points || 0) + points;
    this.customersMap.set(id, customer);
    return customer;
  }

  // Coupon methods
  async getCoupons(): Promise<Coupon[]> {
    return Array.from(this.couponsMap.values());
  }

  async getCoupon(id: number): Promise<Coupon | undefined> {
    return this.couponsMap.get(id);
  }

  async getCouponByCode(code: string): Promise<Coupon | undefined> {
    return Array.from(this.couponsMap.values()).find(coupon => coupon.code === code);
  }

  async createCoupon(insertCoupon: InsertCoupon): Promise<Coupon> {
    const id = this.currentCouponId++;
    const coupon: Coupon = { 
      ...insertCoupon, 
      id,
      usedCount: 0,
      createdAt: new Date()
    };
    this.couponsMap.set(id, coupon);
    return coupon;
  }

  async updateCoupon(id: number, updateData: Partial<InsertCoupon>): Promise<Coupon> {
    const existingCoupon = this.couponsMap.get(id);
    if (!existingCoupon) {
      throw new Error(`Coupon with id ${id} not found`);
    }
    const updatedCoupon: Coupon = {
      ...existingCoupon,
      ...updateData,
    };
    this.couponsMap.set(id, updatedCoupon);
    return updatedCoupon;
  }

  async useCoupon(id: number): Promise<Coupon> {
    const coupon = this.couponsMap.get(id);
    if (!coupon) {
      throw new Error(`Coupon with id ${id} not found`);
    }
    coupon.usedCount = (coupon.usedCount || 0) + 1;
    this.couponsMap.set(id, coupon);
    return coupon;
  }

  async deleteCoupon(id: number): Promise<boolean> {
    return this.couponsMap.delete(id);
  }

  // Analytics methods
  async getAnalytics(limit: number = 100): Promise<Analytics[]> {
    return Array.from(this.analyticsMap.values()).slice(0, limit);
  }

  async createAnalyticsEvent(insertAnalytics: InsertAnalytics): Promise<Analytics> {
    const id = this.currentAnalyticsId++;
    const analytics: Analytics = { 
      ...insertAnalytics, 
      id,
      createdAt: new Date()
    };
    this.analyticsMap.set(id, analytics);
    return analytics;
  }

  async getAnalyticsByEvent(event: string, limit: number = 100): Promise<Analytics[]> {
    return Array.from(this.analyticsMap.values())
      .filter(analytics => analytics.event === event)
      .slice(0, limit);
  }

  // Notification methods
  async getNotifications(): Promise<Notification[]> {
    return Array.from(this.notificationsMap.values());
  }

  async getUnreadNotifications(): Promise<Notification[]> {
    return Array.from(this.notificationsMap.values()).filter(notif => !notif.isRead);
  }

  async createNotification(insertNotification: InsertNotification): Promise<Notification> {
    const id = this.currentNotificationId++;
    const notification: Notification = { 
      ...insertNotification, 
      id,
      isRead: false,
      createdAt: new Date()
    };
    this.notificationsMap.set(id, notification);
    return notification;
  }

  async markNotificationAsRead(id: number): Promise<Notification> {
    const notification = this.notificationsMap.get(id);
    if (!notification) {
      throw new Error(`Notification with id ${id} not found`);
    }
    notification.isRead = true;
    this.notificationsMap.set(id, notification);
    return notification;
  }

  async deleteNotification(id: number): Promise<boolean> {
    return this.notificationsMap.delete(id);
  }

  // Review methods
  async getReviews(): Promise<Review[]> {
    return Array.from(this.reviewsMap.values());
  }

  async getApprovedReviews(): Promise<Review[]> {
    return Array.from(this.reviewsMap.values()).filter(review => review.isApproved);
  }

  async getMenuItemReviews(menuItemId: number): Promise<Review[]> {
    return Array.from(this.reviewsMap.values()).filter(review => review.menuItemId === menuItemId);
  }

  async createReview(insertReview: InsertReview): Promise<Review> {
    const id = this.currentReviewId++;
    const review: Review = { 
      ...insertReview, 
      id,
      isApproved: false,
      isPublic: true,
      createdAt: new Date()
    };
    this.reviewsMap.set(id, review);
    return review;
  }

  async updateReview(id: number, updateData: Partial<InsertReview>): Promise<Review> {
    const existingReview = this.reviewsMap.get(id);
    if (!existingReview) {
      throw new Error(`Review with id ${id} not found`);
    }
    const updatedReview: Review = {
      ...existingReview,
      ...updateData,
    };
    this.reviewsMap.set(id, updatedReview);
    return updatedReview;
  }

  async approveReview(id: number): Promise<Review> {
    return this.updateReview(id, { isApproved: true });
  }

  async deleteReview(id: number): Promise<boolean> {
    return this.reviewsMap.delete(id);
  }
}

export const storage: IStorage = new MemStorage();