import crypto from 'crypto';
import { Payment, InsertPayment } from '@shared/schema';

interface PaymentRequest {
  orderId: number;
  amount: string;
  customerName: string;
  customerEmail?: string;
  customerPhone?: string;
  items: Array<{
    name: string;
    price: string;
    quantity: number;
  }>;
}

interface PaymentResponse {
  success: boolean;
  paymentUrl?: string;
  token?: string;
  errorMessage?: string;
}

export class PayTRService {
  private merchantId: string;
  private merchantKey: string;
  private merchantSalt: string;
  private baseUrl: string;

  constructor() {
    this.merchantId = process.env.PAYTR_MERCHANT_ID || 'test_merchant';
    this.merchantKey = process.env.PAYTR_MERCHANT_KEY || 'test_key';
    this.merchantSalt = process.env.PAYTR_MERCHANT_SALT || 'test_salt';
    this.baseUrl = 'https://www.paytr.com/odeme/api/get-token';
  }

  async createPayment(request: PaymentRequest): Promise<PaymentResponse> {
    try {
      // PayTR için gerekli parametreler
      const params = {
        merchant_id: this.merchantId,
        user_ip: '127.0.0.1', // Gerçek IP adresi
        merchant_oid: `ORDER_${request.orderId}_${Date.now()}`,
        email: request.customerEmail || 'customer@example.com',
        payment_amount: Math.round(parseFloat(request.amount) * 100), // Kuruş cinsinden
        paytr_token: this.generatePayTRToken(request),
        user_basket: this.formatBasket(request.items),
        debug_on: 1, // Test modu
        no_installment: 0,
        max_installment: 0,
        user_name: request.customerName,
        user_address: 'Müşteri Adresi',
        user_phone: request.customerPhone || '5551234567',
        merchant_ok_url: `${process.env.BASE_URL || 'http://localhost:5000'}/api/payment/paytr/success`,
        merchant_fail_url: `${process.env.BASE_URL || 'http://localhost:5000'}/api/payment/paytr/fail`,
        timeout_limit: 30,
        currency: 'TL'
      };

      // API anahtarları olmadan mock response döndür
      if (this.merchantId === 'test_merchant') {
        return {
          success: true,
          paymentUrl: `${process.env.BASE_URL || 'http://localhost:5000'}/mock-payment?provider=paytr&order=${request.orderId}`,
          token: `mock_paytr_token_${Date.now()}`
        };
      }

      // Gerçek PayTR API çağrısı burada yapılacak
      const response = await fetch(this.baseUrl, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: new URLSearchParams(params).toString()
      });

      const result = await response.json();

      if (result.status === 'success') {
        return {
          success: true,
          paymentUrl: result.token,
          token: result.token
        };
      } else {
        return {
          success: false,
          errorMessage: result.reason || 'PayTR ödeme hatası'
        };
      }

    } catch (error) {
      return {
        success: false,
        errorMessage: `PayTR bağlantı hatası: ${error instanceof Error ? error.message : 'Bilinmeyen hata'}`
      };
    }
  }

  private generatePayTRToken(request: PaymentRequest): string {
    const hashStr = `${this.merchantId}${request.customerEmail || 'customer@example.com'}ORDER_${request.orderId}_${Date.now()}${Math.round(parseFloat(request.amount) * 100)}TL127.0.0.1${this.merchantSalt}`;
    return crypto.createHmac('sha256', this.merchantKey).update(hashStr).digest('base64');
  }

  private formatBasket(items: Array<{name: string, price: string, quantity: number}>): string {
    return items.map(item => 
      `["${item.name}", "${item.price}", ${item.quantity}]`
    ).join(',');
  }

  verifyCallback(params: any): boolean {
    const hash = crypto.createHmac('sha256', this.merchantKey)
      .update(`${params.merchant_oid}${this.merchantSalt}${params.status}${params.total_amount}`)
      .digest('base64');
    
    return hash === params.hash;
  }
}

export class IyzicoService {
  private apiKey: string;
  private secretKey: string;
  private baseUrl: string;

  constructor() {
    this.apiKey = process.env.IYZICO_API_KEY || 'test_api_key';
    this.secretKey = process.env.IYZICO_SECRET_KEY || 'test_secret_key';
    this.baseUrl = 'https://sandbox-api.iyzipay.com'; // Test için sandbox
  }

  async createPayment(request: PaymentRequest): Promise<PaymentResponse> {
    try {
      const requestData = {
        locale: 'tr',
        conversationId: `conv_${request.orderId}_${Date.now()}`,
        price: request.amount,
        paidPrice: request.amount,
        currency: 'TRY',
        installment: 1,
        basketId: `basket_${request.orderId}`,
        paymentChannel: 'WEB',
        paymentGroup: 'PRODUCT',
        callbackUrl: `${process.env.BASE_URL || 'http://localhost:5000'}/api/payment/iyzico/callback`,
        enabledInstallments: [1],
        buyer: {
          id: `buyer_${Date.now()}`,
          name: request.customerName.split(' ')[0] || 'Ad',
          surname: request.customerName.split(' ')[1] || 'Soyad',
          gsmNumber: request.customerPhone || '+905551234567',
          email: request.customerEmail || 'customer@example.com',
          identityNumber: '11111111111',
          lastLoginDate: new Date().toISOString(),
          registrationDate: new Date().toISOString(),
          registrationAddress: 'Müşteri Adresi',
          ip: '127.0.0.1',
          city: 'Istanbul',
          country: 'Turkey',
          zipCode: '34732'
        },
        shippingAddress: {
          contactName: request.customerName,
          city: 'Istanbul',
          country: 'Turkey',
          address: 'Müşteri Adresi',
          zipCode: '34732'
        },
        billingAddress: {
          contactName: request.customerName,
          city: 'Istanbul',
          country: 'Turkey',
          address: 'Müşteri Adresi',
          zipCode: '34732'
        },
        basketItems: request.items.map((item, index) => ({
          id: `item_${index}`,
          name: item.name,
          category1: 'Food',
          itemType: 'PHYSICAL',
          price: (parseFloat(item.price) * item.quantity).toFixed(2)
        }))
      };

      // API anahtarları olmadan mock response döndür
      if (this.apiKey === 'test_api_key') {
        return {
          success: true,
          paymentUrl: `${process.env.BASE_URL || 'http://localhost:5000'}/mock-payment?provider=iyzico&order=${request.orderId}`,
          token: `mock_iyzico_token_${Date.now()}`
        };
      }

      // Gerçek Iyzico API çağrısı burada yapılacak
      const authHeader = this.generateIyzicoAuth(requestData);
      
      const response = await fetch(`${this.baseUrl}/payment/iyzipos/checkoutform/initialize/auth/ecom`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': authHeader,
          'x-iyzi-rnd': crypto.randomBytes(16).toString('hex')
        },
        body: JSON.stringify(requestData)
      });

      const result = await response.json();

      if (result.status === 'success') {
        return {
          success: true,
          paymentUrl: result.checkoutFormContent,
          token: result.token
        };
      } else {
        return {
          success: false,
          errorMessage: result.errorMessage || 'Iyzico ödeme hatası'
        };
      }

    } catch (error) {
      return {
        success: false,
        errorMessage: `Iyzico bağlantı hatası: ${error instanceof Error ? error.message : 'Bilinmeyen hata'}`
      };
    }
  }

  private generateIyzicoAuth(requestData: any): string {
    const randomKey = crypto.randomBytes(16).toString('hex');
    const dataString = JSON.stringify(requestData);
    const hashData = `${this.apiKey}${randomKey}${this.secretKey}${dataString}`;
    const hash = crypto.createHmac('sha1', this.secretKey).update(hashData).digest('base64');
    
    return `IYZWS ${this.apiKey}:${hash}`;
  }

  verifyCallback(token: string): boolean {
    // Iyzico callback doğrulama mantığı
    return token && token.length > 0;
  }
}

export class PaymentService {
  private paytr: PayTRService;
  private iyzico: IyzicoService;

  constructor() {
    this.paytr = new PayTRService();
    this.iyzico = new IyzicoService();
  }

  async processPayment(provider: 'paytr' | 'iyzico', request: PaymentRequest): Promise<PaymentResponse> {
    switch (provider) {
      case 'paytr':
        return await this.paytr.createPayment(request);
      case 'iyzico':
        return await this.iyzico.createPayment(request);
      default:
        return {
          success: false,
          errorMessage: 'Geçersiz ödeme sağlayıcısı'
        };
    }
  }

  verifyPayment(provider: 'paytr' | 'iyzico', params: any): boolean {
    switch (provider) {
      case 'paytr':
        return this.paytr.verifyCallback(params);
      case 'iyzico':
        return this.iyzico.verifyCallback(params.token);
      default:
        return false;
    }
  }
}