import { db } from "./db-mysql";
import { 
  users, 
  categories, 
  menuItems, 
  restaurants, 
  themeSettings, 
  seoSettings,
  images,
  menuItemImages,
  orders,
  orderItems,
  customers,
  coupons,
  analytics,
  notifications,
  reviews,
  payments
} from "@shared/schema-mysql";
import { scrypt, randomBytes, timingSafeEqual } from "crypto";
import { promisify } from "util";

const scryptAsync = promisify(scrypt);

async function hashPassword(password: string) {
  const salt = randomBytes(16).toString("hex");
  const buf = (await scryptAsync(password, salt, 64)) as Buffer;
  return `${buf.toString("hex")}.${salt}`;
}

export async function runMigrations() {
  console.log("Starting MySQL database setup and migration...");
  
  try {
    // Create tables if they don't exist (MySQL will ignore if exists)
    await createTables();
    
    // Insert default data
    await insertDefaultData();
    
    console.log("✓ MySQL database migration completed successfully");
  } catch (error) {
    console.error("MySQL migration failed:", error);
    throw error;
  }
}

async function createTables() {
  console.log("Creating MySQL tables...");
  
  // Users table
  await db.execute(`
    CREATE TABLE IF NOT EXISTS users (
      id INT AUTO_INCREMENT PRIMARY KEY,
      username VARCHAR(255) NOT NULL UNIQUE,
      password TEXT NOT NULL
    )
  `);

  // Categories table
  await db.execute(`
    CREATE TABLE IF NOT EXISTS categories (
      id INT AUTO_INCREMENT PRIMARY KEY,
      name VARCHAR(255) NOT NULL,
      slug VARCHAR(255) NOT NULL UNIQUE,
      \`order\` INT NOT NULL
    )
  `);

  // Menu items table
  await db.execute(`
    CREATE TABLE IF NOT EXISTS menu_items (
      id INT AUTO_INCREMENT PRIMARY KEY,
      name VARCHAR(255) NOT NULL,
      description TEXT,
      price VARCHAR(20) NOT NULL,
      category_id INT NOT NULL,
      image TEXT,
      images TEXT,
      is_active BOOLEAN DEFAULT TRUE,
      is_vegetarian BOOLEAN DEFAULT FALSE,
      is_vegan BOOLEAN DEFAULT FALSE,
      is_gluten_free BOOLEAN DEFAULT FALSE,
      is_spicy BOOLEAN DEFAULT FALSE,
      is_chef_special BOOLEAN DEFAULT FALSE,
      is_monthly_special BOOLEAN DEFAULT FALSE,
      is_campaign BOOLEAN DEFAULT FALSE,
      campaign_price VARCHAR(20),
      sort_order INT DEFAULT 0,
      preparation_time VARCHAR(50),
      allergens TEXT,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )
  `);

  // Restaurants table
  await db.execute(`
    CREATE TABLE IF NOT EXISTS restaurants (
      id INT AUTO_INCREMENT PRIMARY KEY,
      name VARCHAR(255) NOT NULL,
      description TEXT,
      address TEXT,
      phone VARCHAR(20),
      email VARCHAR(255),
      website VARCHAR(255),
      logo TEXT,
      is_active BOOLEAN DEFAULT TRUE,
      enable_payments BOOLEAN DEFAULT FALSE,
      enable_paytr BOOLEAN DEFAULT FALSE,
      enable_iyzico BOOLEAN DEFAULT FALSE,
      enable_cash_payment BOOLEAN DEFAULT TRUE,
      enable_order_system BOOLEAN DEFAULT FALSE,
      paytr_commission_rate VARCHAR(10) DEFAULT '2.9',
      iyzico_commission_rate VARCHAR(10) DEFAULT '2.7',
      fixed_fee_paytr VARCHAR(10) DEFAULT '0.25',
      fixed_fee_iyzico VARCHAR(10) DEFAULT '0.30',
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )
  `);

  // Theme settings table
  await db.execute(`
    CREATE TABLE IF NOT EXISTS theme_settings (
      id INT AUTO_INCREMENT PRIMARY KEY,
      name VARCHAR(255) NOT NULL,
      primary_color VARCHAR(7) NOT NULL,
      secondary_color VARCHAR(7) NOT NULL,
      background_color VARCHAR(7) NOT NULL,
      text_color VARCHAR(7) NOT NULL,
      font_family VARCHAR(100) NOT NULL,
      is_active BOOLEAN DEFAULT FALSE,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )
  `);

  // SEO settings table
  await db.execute(`
    CREATE TABLE IF NOT EXISTS seo_settings (
      id INT AUTO_INCREMENT PRIMARY KEY,
      page_title VARCHAR(255),
      meta_description TEXT,
      meta_keywords TEXT,
      og_title VARCHAR(255),
      og_description TEXT,
      og_image TEXT,
      twitter_title VARCHAR(255),
      twitter_description TEXT,
      twitter_image TEXT,
      canonical_url TEXT,
      robots_content TEXT,
      structured_data TEXT,
      is_active BOOLEAN DEFAULT TRUE,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )
  `);

  // Create remaining e-commerce tables
  await createEcommerceTables();

  console.log("✓ All MySQL tables created successfully");
}

async function createEcommerceTables() {
  // Images table
  await db.execute(`
    CREATE TABLE IF NOT EXISTS images (
      id INT AUTO_INCREMENT PRIMARY KEY,
      url TEXT NOT NULL,
      alt VARCHAR(255),
      menu_item_id INT,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )
  `);

  // Menu item images table
  await db.execute(`
    CREATE TABLE IF NOT EXISTS menu_item_images (
      id INT AUTO_INCREMENT PRIMARY KEY,
      menu_item_id INT NOT NULL,
      image_id INT NOT NULL,
      \`order\` INT DEFAULT 0
    )
  `);

  // Orders table
  await db.execute(`
    CREATE TABLE IF NOT EXISTS orders (
      id INT AUTO_INCREMENT PRIMARY KEY,
      order_number VARCHAR(50) NOT NULL UNIQUE,
      table_number VARCHAR(10),
      customer_name VARCHAR(255),
      customer_phone VARCHAR(20),
      status VARCHAR(20) NOT NULL DEFAULT 'pending',
      total_amount VARCHAR(20) NOT NULL,
      notes TEXT,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )
  `);

  // Order items table
  await db.execute(`
    CREATE TABLE IF NOT EXISTS order_items (
      id INT AUTO_INCREMENT PRIMARY KEY,
      order_id INT NOT NULL,
      menu_item_id INT NOT NULL,
      quantity INT NOT NULL,
      price VARCHAR(20) NOT NULL,
      notes TEXT
    )
  `);

  // Customers table
  await db.execute(`
    CREATE TABLE IF NOT EXISTS customers (
      id INT AUTO_INCREMENT PRIMARY KEY,
      name VARCHAR(255) NOT NULL,
      phone VARCHAR(20) UNIQUE,
      email VARCHAR(255) UNIQUE,
      address TEXT,
      points INT DEFAULT 0,
      total_spent VARCHAR(20) DEFAULT '0',
      total_orders INT DEFAULT 0,
      visits INT DEFAULT 0,
      last_visit TIMESTAMP,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )
  `);

  // Coupons table
  await db.execute(`
    CREATE TABLE IF NOT EXISTS coupons (
      id INT AUTO_INCREMENT PRIMARY KEY,
      code VARCHAR(50) NOT NULL UNIQUE,
      name VARCHAR(255) NOT NULL,
      description TEXT,
      discount_type VARCHAR(20) NOT NULL,
      discount_value VARCHAR(20) NOT NULL,
      min_order_amount VARCHAR(20) DEFAULT '0',
      max_uses INT DEFAULT 1,
      used_count INT DEFAULT 0,
      is_active BOOLEAN DEFAULT TRUE,
      expires_at TIMESTAMP,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )
  `);

  // Analytics table
  await db.execute(`
    CREATE TABLE IF NOT EXISTS analytics (
      id INT AUTO_INCREMENT PRIMARY KEY,
      event VARCHAR(100) NOT NULL,
      data TEXT,
      ip_address VARCHAR(45),
      user_agent TEXT,
      timestamp TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )
  `);

  // Notifications table
  await db.execute(`
    CREATE TABLE IF NOT EXISTS notifications (
      id INT AUTO_INCREMENT PRIMARY KEY,
      title VARCHAR(255) NOT NULL,
      message TEXT NOT NULL,
      type VARCHAR(50) NOT NULL,
      is_read BOOLEAN DEFAULT FALSE,
      target_audience VARCHAR(50) DEFAULT 'admin',
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )
  `);

  // Reviews table
  await db.execute(`
    CREATE TABLE IF NOT EXISTS reviews (
      id INT AUTO_INCREMENT PRIMARY KEY,
      order_id INT,
      customer_name VARCHAR(255) NOT NULL,
      customer_email VARCHAR(255),
      rating INT NOT NULL,
      comment TEXT,
      menu_item_id INT,
      is_approved BOOLEAN DEFAULT FALSE,
      is_public BOOLEAN DEFAULT TRUE,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )
  `);

  // Payments table
  await db.execute(`
    CREATE TABLE IF NOT EXISTS payments (
      id INT AUTO_INCREMENT PRIMARY KEY,
      order_id INT NOT NULL,
      payment_method VARCHAR(50) NOT NULL,
      payment_provider VARCHAR(50),
      amount VARCHAR(20) NOT NULL,
      currency VARCHAR(3) DEFAULT 'TRY',
      status VARCHAR(20) NOT NULL DEFAULT 'pending',
      transaction_id VARCHAR(100),
      payment_token VARCHAR(255),
      payment_url TEXT,
      error_message TEXT,
      metadata TEXT,
      paid_at TIMESTAMP,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )
  `);
}

async function insertDefaultData() {
  console.log("Inserting default data...");

  // Check if admin user exists
  const existingUser = await db.select().from(users).where(eq(users.username, 'admin')).limit(1);
  
  if (existingUser.length === 0) {
    const hashedPassword = await hashPassword("admin123");
    await db.insert(users).values({
      username: "admin",
      password: hashedPassword,
    });
    console.log("✓ Admin user created");
  }

  // Check if restaurant info exists
  const existingRestaurant = await db.select().from(restaurants).limit(1);
  
  if (existingRestaurant.length === 0) {
    await db.insert(restaurants).values({
      name: "Jouska Cafe & Restaurant",
      description: "Modern Türk mutfağı ve dünya lezzetlerinin buluştuğu özel mekan",
      address: "İstanbul, Türkiye",
      phone: "+90 212 123 4567",
      email: "info@jouska.com",
      website: "https://jouska.com",
      isActive: true,
      enablePayments: false,
      enableCashPayment: true,
      enableOrderSystem: false,
    });
    console.log("✓ Restaurant information created");
  }

  // Check if theme settings exist
  const existingTheme = await db.select().from(themeSettings).limit(1);
  
  if (existingTheme.length === 0) {
    await db.insert(themeSettings).values({
      name: "Varsayılan Tema",
      primaryColor: "#8B5CF6",
      secondaryColor: "#06B6D4",
      backgroundColor: "#FFFFFF",
      textColor: "#1F2937",
      fontFamily: "Inter",
      isActive: true,
    });
    console.log("✓ Default theme created");
  }

  // Check if SEO settings exist
  const existingSeo = await db.select().from(seoSettings).limit(1);
  
  if (existingSeo.length === 0) {
    await db.insert(seoSettings).values({
      pageTitle: "Jouska Cafe & Restaurant - QR Menü",
      metaDescription: "Modern Türk mutfağı ve dünya lezzetlerinin buluştuğu Jouska Cafe & Restaurant. QR menümüzden sipariş verin.",
      metaKeywords: "restoran, cafe, türk mutfağı, qr menü, online sipariş",
      ogTitle: "Jouska Cafe & Restaurant",
      ogDescription: "Modern Türk mutfağı ve dünya lezzetlerinin buluştuğu özel mekan",
      isActive: true,
    });
    console.log("✓ SEO settings created");
  }

  console.log("✓ All default data created successfully");
}

// Auto-run migrations when this file is executed directly
if (import.meta.url === `file://${process.argv[1]}`) {
  runMigrations()
    .then(() => {
      console.log("Migration completed successfully");
      process.exit(0);
    })
    .catch((error) => {
      console.error("Migration failed:", error);
      process.exit(1);
    });
}