import React, { useEffect, useState } from "react";
import { useRoute, Link } from "wouter";
import { Header } from "@/components/layout/header";
import { Footer } from "@/components/layout/footer";
import { MenuItemCard } from "@/components/menu/menu-item";
import { Button } from "@/components/ui/button";
import { ChevronLeft, ArrowLeft } from "lucide-react";
import { useQuery } from "@tanstack/react-query";
import { useMenuItems } from "@/hooks/use-menu-items";
import { Restaurant, Category } from "@shared/schema";
import { SectionTitle } from "@/components/ui/section-title";
import { RelatedMenuItems } from "@/components/related-menu-items";

export default function CategoryPage() {
  const [, params] = useRoute("/category/:slug");
  const slug = params?.slug;
  
  const [categoryData, setCategoryData] = useState<Category | null>(null);
  const [searchQuery, setSearchQuery] = useState<string>("");
  
  // Fetch restaurant information
  const restaurantQuery = useQuery<Restaurant>({ 
    queryKey: ['/api/restaurant'] 
  });
  
  // Fetch all categories
  const categoriesQuery = useQuery<Category[]>({ 
    queryKey: ['/api/categories'] 
  });
  
  // Fetch all menu items
  const { data: menuItems } = useMenuItems();
  
  // Set current category data
  useEffect(() => {
    if (categoriesQuery.data && slug) {
      const currentCategory = categoriesQuery.data.find(cat => cat.slug === slug);
      setCategoryData(currentCategory || null);
    }
  }, [categoriesQuery.data, slug]);
  
  // Filter menu items by category
  const categoryItems = menuItems?.filter(
    item => categoryData && item.categoryId === categoryData.id
  ) || [];
  
  // Loading state
  if (!categoryData || !menuItems) {
    return (
      <div className="min-h-screen flex flex-col">
        <Header 
          restaurantInfo={restaurantQuery.data}
          onQrButtonClick={() => {}}
          searchQuery={searchQuery}
          setSearchQuery={setSearchQuery}
        />
        <main className="flex-grow container mx-auto px-4 py-12 flex justify-center items-center">
          <div className="text-center">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-primary mx-auto mb-4"></div>
            <h2 className="text-xl font-medium text-gray-600">Yükleniyor...</h2>
          </div>
        </main>
        <Footer restaurantInfo={restaurantQuery.data} />
      </div>
    );
  }
  
  return (
    <div className="min-h-screen flex flex-col">
      <Header 
        restaurantInfo={restaurantQuery.data}
        onQrButtonClick={() => {}}
        searchQuery={searchQuery}
        setSearchQuery={setSearchQuery}
      />
      
      <div className="bg-gradient-to-r from-primary/95 to-primary shadow-md py-3">
        <div className="container mx-auto px-4">
          <Link href="/">
            <Button 
              variant="ghost" 
              className="text-white hover:bg-white/10"
            >
              <ArrowLeft className="h-4 w-4 mr-2" />
              Tüm Menü
            </Button>
          </Link>
        </div>
      </div>
      
      <main className="flex-grow container mx-auto px-4 py-6">
        {/* Category Header */}
        <div className="mb-8 bg-gradient-to-br from-primary/5 to-accent/5 rounded-xl p-6 shadow-sm border border-primary/10">
          <SectionTitle 
            title={categoryData.name}
            subtitle={`${categoryItems.length} ürün bulundu`}
            size="lg"
            decorative={true}
            centered={true}
          />
        </div>
        
        {/* Category Items */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
          {categoryItems.length > 0 ? (
            categoryItems.map((item) => (
              <MenuItemCard key={item.id} item={item} />
            ))
          ) : (
            <div className="col-span-2 py-12 text-center">
              <div className="inline-block p-6 rounded-lg bg-primary/5 text-primary">
                <svg xmlns="http://www.w3.org/2000/svg" className="h-12 w-12 mx-auto mb-4 opacity-50" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1} d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                </svg>
                <p className="text-lg font-medium mb-2">Bu kategoride henüz ürün bulunmamaktadır</p>
                <p className="text-sm opacity-75">Çok yakında yeni ürünler eklenecektir.</p>
              </div>
            </div>
          )}
        </div>
        
        {/* İlgili menü öğeleri */}
        {categoryData && categoryItems.length > 0 && (
          <RelatedMenuItems 
            currentCategoryId={categoryData.id}
            excludeMenuItemIds={categoryItems.map(item => item.id)}
          />
        )}
      </main>
      
      <Footer restaurantInfo={restaurantQuery.data} />
    </div>
  );
}