import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { useToast } from "./use-toast";
import { Image } from "@shared/schema";

// Tüm görselleri getir
export function useImages() {
  return useQuery<Image[]>({
    queryKey: ['/api/images'],
    refetchOnWindowFocus: false,
    refetchOnMount: false,
    refetchOnReconnect: false,
    staleTime: 300000 // 5 dakika boyunca verileri taze kabul et
  });
}

// Tek bir görseli getir
export function useImage(id: number) {
  return useQuery<Image>({
    queryKey: ['/api/images', id],
    enabled: !!id,
    refetchOnWindowFocus: false,
    staleTime: 300000
  });
}

// Menü öğesine ait görselleri getir
export function useMenuItemImages(menuItemId: number) {
  return useQuery<Image[]>({
    queryKey: ['/api/menu-items', menuItemId, 'images'],
    queryFn: async () => {
      const res = await fetch(`/api/menu-items/${menuItemId}/images`);
      if (!res.ok) {
        throw new Error('Failed to fetch menu item images');
      }
      return res.json();
    },
    enabled: !!menuItemId,
    refetchOnWindowFocus: false,
    refetchOnMount: false,
    staleTime: 300000 // 5 dakika boyunca verileri taze kabul et
  });
}

// Görsel oluştur
export function useCreateImage() {
  const { toast } = useToast();
  
  return useMutation({
    mutationFn: async (imageData: any) => {
      const res = await apiRequest("/api/images", {
        method: "POST",
        body: JSON.stringify(imageData)
      });
      return await res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/images'] });
      toast({
        title: "Görsel eklendi",
        description: "Yeni görsel başarıyla eklendi.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Görsel eklenemedi",
        description: error.message || "Görsel eklenirken bir hata oluştu.",
        variant: "destructive"
      });
    }
  });
}

// Görsel güncelle
export function useUpdateImage() {
  const { toast } = useToast();
  
  return useMutation({
    mutationFn: async ({ id, data }: { id: number; data: any }) => {
      const res = await apiRequest(`/api/images/${id}`, { 
        method: "PUT",
        body: JSON.stringify(data)
      });
      return await res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/images'] });
      toast({
        title: "Görsel güncellendi",
        description: "Görsel başarıyla güncellendi.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Görsel güncellenemedi",
        description: error.message || "Görsel güncellenirken bir hata oluştu.",
        variant: "destructive"
      });
    }
  });
}

// Görsel sil
export function useDeleteImage() {
  const { toast } = useToast();
  
  return useMutation({
    mutationFn: async (id: number) => {
      await apiRequest(`/api/images/${id}`, { 
        method: "DELETE" 
      });
      return true;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/images'] });
      toast({
        title: "Görsel silindi",
        description: "Görsel başarıyla silindi.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Görsel silinemedi",
        description: error.message || "Görsel silinirken bir hata oluştu.",
        variant: "destructive"
      });
    }
  });
}

// Görsel menü öğesine ekle
export function useAddImageToMenuItem() {
  const { toast } = useToast();
  
  return useMutation({
    mutationFn: async ({ menuItemId, imageId, order = 0 }: { menuItemId: number; imageId: number; order?: number }) => {
      const res = await apiRequest(`/api/menu-items/${menuItemId}/images/${imageId}`, {
        method: "POST",
        body: JSON.stringify({ order })
      });
      return await res.json();
    },
    onSuccess: (_, variables) => {
      queryClient.invalidateQueries({ queryKey: ['/api/menu-items', variables.menuItemId, 'images'] });
      toast({
        title: "Görsel eklendi",
        description: "Görsel başarıyla menü öğesine eklendi.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Görsel eklenemedi",
        description: error.message || "Görsel menü öğesine eklenirken bir hata oluştu.",
        variant: "destructive"
      });
    }
  });
}

// Görseli menü öğesinden kaldır
export function useRemoveImageFromMenuItem() {
  const { toast } = useToast();
  
  return useMutation({
    mutationFn: async ({ menuItemId, imageId }: { menuItemId: number; imageId: number }) => {
      await apiRequest(`/api/menu-items/${menuItemId}/images/${imageId}`, {
        method: "DELETE"
      });
      return true;
    },
    onSuccess: (_, variables) => {
      queryClient.invalidateQueries({ queryKey: ['/api/menu-items', variables.menuItemId, 'images'] });
      toast({
        title: "Görsel kaldırıldı",
        description: "Görsel başarıyla menü öğesinden kaldırıldı.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Görsel kaldırılamadı",
        description: error.message || "Görsel menü öğesinden kaldırılırken bir hata oluştu.",
        variant: "destructive"
      });
    }
  });
}