import { useState, useRef, useEffect } from "react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Avatar, AvatarFallback } from "@/components/ui/avatar";
import { 
  MessageCircle, 
  Send, 
  X, 
  Search,
  ChefHat,
  Coffee,
  Utensils,
  Heart,
  Clock,
  Star
} from "lucide-react";
import { motion, AnimatePresence } from "framer-motion";
import { useLocation } from "wouter";
import { useCategories } from "@/hooks/use-categories";
import { useMenuItems } from "@/hooks/use-menu-items";

interface Message {
  id: string;
  text: string;
  isBot: boolean;
  timestamp: Date;
  quickReplies?: string[];
}

interface QuickReply {
  text: string;
  action: string;
  icon?: React.ReactNode;
}

interface WelcomeChatbotProps {
  isOpen: boolean;
  onClose: () => void;
  onSearch: (query: string) => void;
  onCategorySelect: (categorySlug: string) => void;
  restaurantName?: string;
}

export function WelcomeChatbot({ 
  isOpen, 
  onClose, 
  onSearch, 
  onCategorySelect,
  restaurantName = "Jouska Cafe & Restaurant"
}: WelcomeChatbotProps) {
  const [location, setLocation] = useLocation();
  const [messages, setMessages] = useState<Message[]>([]);
  const [inputValue, setInputValue] = useState("");
  const [isTyping, setIsTyping] = useState(false);
  const messagesEndRef = useRef<HTMLDivElement>(null);
  
  // Fetch data for categories and menu items
  const { data: categories } = useCategories();
  const { data: menuItems } = useMenuItems();

  // Function to get menu items for a specific category
  const fetchCategoryItems = async (categorySlug: string) => {
    if (!categories || !menuItems) return [];
    
    const category = categories.find(cat => cat.slug === categorySlug);
    if (!category) return [];
    
    const categoryItems = menuItems.filter(item => 
      item.categoryId === category.id && item.isActive
    );
    
    return categoryItems;
  };

  const quickReplies: QuickReply[] = [
    { text: "Menüyü görüntüle", action: "show_menu", icon: <Utensils className="w-4 h-4" /> },
    { text: "Kahvaltılar", action: "category_kahvaltilar", icon: <Coffee className="w-4 h-4" /> },
    { text: "Ana Yemekler", action: "category_ana-yemekler", icon: <ChefHat className="w-4 h-4" /> },
    { text: "Popüler Ürünler", action: "popular", icon: <Star className="w-4 h-4" /> },
    { text: "Vejetaryen", action: "search_vejetaryen", icon: <Heart className="w-4 h-4" /> },
    { text: "Google'da Yorumla", action: "google_review", icon: <Star className="w-4 h-4" /> }
  ];

  // Admin panelinden ayarları yükle
  const [chatbotConfig, setChatbotConfig] = useState<any>(null);

  useEffect(() => {
    const loadChatbotConfig = () => {
      try {
        const savedConfig = localStorage.getItem('chatbotConfig');
        
        // Check if we need to add default Q&A pairs to existing config
        if (savedConfig) {
          const config = JSON.parse(savedConfig);
          
          // Ensure autoOpen exists in config, default to true if missing
          if (config.autoOpen === undefined) {
            config.autoOpen = true;
            config.openDelay = config.openDelay || 1500;
            localStorage.setItem('chatbotConfig', JSON.stringify(config));
          }
          
          // If no Q&A pairs exist, add defaults
          if (!config.qnaPairs || config.qnaPairs.length === 0) {
            const defaultQnaPairs = [
              {
                id: "q1",
                question: "Çay var mı?",
                answer: "Evet, çeşitli çay seçeneklerimiz bulunmaktadır. Türk çayı, yeşil çay, bitki çayları ve özel karışımlarımız menümüzde yer almaktadır.",
                keywords: ["çay", "tea", "içecek"]
              },
              {
                id: "q2", 
                question: "Kahve çeşitleriniz neler?",
                answer: "Türk kahvesi, espresso, cappuccino, latte, americano ve özel soğuk kahve çeşitlerimiz bulunmaktadır. Taze çekilmiş kahvelerimizle size en iyi lezzeti sunuyoruz.",
                keywords: ["kahve", "coffee", "espresso", "latte", "cappuccino"]
              },
              {
                id: "q3",
                question: "Vejetaryen seçenekleriniz var mı?",
                answer: "Evet, geniş vejetaryen menümüz bulunmaktadır. Salatalar, vejetaryen ana yemekler ve tatlılarımız mevcuttur. Menüde 🌱 işaretiyle belirtilmiştir.",
                keywords: ["vejetaryen", "vegetarian", "vegan", "bitki", "salata"]
              },
              {
                id: "q4",
                question: "Rezervasyon yapabilir miyim?",
                answer: "Evet, rezervasyon yapabilirsiniz. Telefon ile +90 XXX XXX XX XX numaramızdan veya online rezervasyon sistemimizden kolayca rezervasyon yapabilirsiniz.",
                keywords: ["rezervasyon", "reservation", "masa", "table"]
              },
              {
                id: "q5",
                question: "Çalışma saatleriniz nedir?",
                answer: "Hafta içi 08:00-23:00, hafta sonu 09:00-24:00 saatleri arasında hizmet vermekteyiz. Özel günlerde saatlerimiz değişebilir.",
                keywords: ["saat", "açık", "kapalı", "çalışma", "hours"]
              },
              {
                id: "q6",
                question: "Paket servis yapıyor musunuz?",
                answer: "Evet, tüm menüdeki ürünlerimiz paket servise uygundur. Telefon ile sipariş verebilir veya online sipariş sistemimizi kullanabilirsiniz.",
                keywords: ["paket", "takeaway", "sipariş", "delivery"]
              }
            ];
            
            config.qnaPairs = defaultQnaPairs;
            localStorage.setItem('chatbotConfig', JSON.stringify(config));
          }
          
          setChatbotConfig({
            ...config,
            qnaPairs: config.qnaPairs || [],
            customerQuestions: config.customerQuestions || []
          });
        } else {
          // Varsayılan ayarlar
          const defaultConfig = {
            isEnabled: true,
            welcomeMessage: `Merhaba, ${restaurantName}'ya hoş geldiniz! 🌟 Size nasıl yardımcı olabilirim? Menüye göz atmak veya arama yaparak istediğiniz lezzetleri daha hızlı bulmanız için buradayım. Ne arıyorsunuz?`,
            botName: "Jouska Asistan",
            quickReplies: quickReplies.map(reply => reply.text),
            qnaPairs: [
              {
                id: "q1",
                question: "Çay var mı?",
                answer: "Evet, çeşitli çay seçeneklerimiz bulunmaktadır. Türk çayı, yeşil çay, bitki çayları ve özel karışımlarımız menümüzde yer almaktadır.",
                keywords: ["çay", "tea", "içecek"]
              },
              {
                id: "q2", 
                question: "Kahve çeşitleriniz neler?",
                answer: "Türk kahvesi, espresso, cappuccino, latte, americano ve özel soğuk kahve çeşitlerimiz bulunmaktadır. Taze çekilmiş kahvelerimizle size en iyi lezzeti sunuyoruz.",
                keywords: ["kahve", "coffee", "espresso", "latte", "cappuccino"]
              },
              {
                id: "q3",
                question: "Vejetaryen seçenekleriniz var mı?",
                answer: "Evet, geniş vejetaryen menümüz bulunmaktadır. Salatalar, vejetaryen ana yemekler ve tatlılarımız mevcuttur. Menüde 🌱 işaretiyle belirtilmiştir.",
                keywords: ["vejetaryen", "vegetarian", "vegan", "bitki", "salata"]
              },
              {
                id: "q4",
                question: "Rezervasyon yapabilir miyim?",
                answer: "Evet, rezervasyon yapabilirsiniz. Telefon ile +90 XXX XXX XX XX numaramızdan veya online rezervasyon sistemimizden kolayca rezervasyon yapabilirsiniz.",
                keywords: ["rezervasyon", "reservation", "masa", "table"]
              },
              {
                id: "q5",
                question: "Çalışma saatleriniz nedir?",
                answer: "Hafta içi 08:00-23:00, hafta sonu 09:00-24:00 saatleri arasında hizmet vermekteyiz. Özel günlerde saatlerimiz değişebilir.",
                keywords: ["saat", "açık", "kapalı", "çalışma", "hours"]
              },
              {
                id: "q6",
                question: "Paket servis yapıyor musunuz?",
                answer: "Evet, tüm menüdeki ürünlerimiz paket servise uygundur. Telefon ile sipariş verebilir veya online sipariş sistemimizi kullanabilirsiniz.",
                keywords: ["paket", "takeaway", "sipariş", "delivery"]
              }
            ]
          };
          setChatbotConfig(defaultConfig);
        }
      } catch (error) {
        console.error('Chatbot config loading error:', error);
        setChatbotConfig({
          isEnabled: true,
          welcomeMessage: `Merhaba, ${restaurantName}'ya hoş geldiniz! 🌟 Size nasıl yardımcı olabilirim?`,
          botName: "Jouska Asistan",
          quickReplies: quickReplies.map(reply => reply.text),
          qnaPairs: []
        });
      }
    };

    loadChatbotConfig();
    
    // Listen for localStorage changes
    const handleStorageChange = () => {
      loadChatbotConfig();
    };
    
    // Custom event listener for same-tab localStorage changes
    const handleCustomStorageChange = () => {
      loadChatbotConfig();
    };
    
    window.addEventListener('storage', handleStorageChange);
    window.addEventListener('chatbotConfigUpdated', handleCustomStorageChange);
    
    return () => {
      window.removeEventListener('storage', handleStorageChange);
      window.removeEventListener('chatbotConfigUpdated', handleCustomStorageChange);
    };
  }, [restaurantName]);

  const welcomeMessage: Message = {
    id: "welcome",
    text: chatbotConfig?.welcomeMessage || `Merhaba, ${restaurantName}'ya hoş geldiniz! 🌟 Size nasıl yardımcı olabilirim? Menüye göz atmak veya arama yaparak istediğiniz lezzetleri daha hızlı bulmanız için buradayım. Ne arıyorsunuz?`,
    isBot: true,
    timestamp: new Date(),
    quickReplies: chatbotConfig?.quickReplies || quickReplies.map(reply => reply.text)
  };

  useEffect(() => {
    if (isOpen && messages.length === 0) {
      setIsTyping(true);
      setTimeout(() => {
        setMessages([welcomeMessage]);
        setIsTyping(false);
      }, 1000);
    }
  }, [isOpen]);

  useEffect(() => {
    scrollToBottom();
  }, [messages, isTyping]);

  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: "smooth" });
  };

  // Q&A eşleştirme fonksiyonu
  const findQnaMatch = (userText: string) => {
    console.log('findQnaMatch called with:', userText);
    console.log('chatbotConfig:', chatbotConfig);
    
    if (!chatbotConfig?.qnaPairs) {
      console.log('No qnaPairs found');
      return null;
    }
    
    console.log('Available Q&A pairs:', chatbotConfig.qnaPairs);
    
    const text = userText.toLowerCase().trim();
    
    // Aktif Q&A çiftlerini kontrol et
    for (const pair of chatbotConfig.qnaPairs) {
      console.log('Checking pair:', pair, 'isActive:', pair.isActive);
      
      if (!pair.isActive) continue;
      
      // Soru ile tam eşleşme
      const questionMatch = pair.question.toLowerCase().includes(text) || text.includes(pair.question.toLowerCase());
      console.log('Question match for', pair.question, ':', questionMatch);
      
      if (questionMatch) {
        console.log('Found question match:', pair);
        return pair;
      }
      
      // Anahtar kelimelerle eşleşme
      for (const keyword of pair.keywords) {
        const keywordMatch = text.includes(keyword.toLowerCase()) || keyword.toLowerCase().includes(text);
        console.log('Keyword match for', keyword, ':', keywordMatch);
        
        if (keywordMatch) {
          console.log('Found keyword match:', pair);
          return pair;
        }
      }
    }
    
    console.log('No match found');
    return null;
  };

  const handleQuickReply = (reply: string) => {
    // Önce Q&A eşleşmesi kontrol et
    const qnaMatch = findQnaMatch(reply);
    
    const quickReplyAction = quickReplies.find(qr => qr.text === reply);
    
    // Kullanıcı mesajını ekle
    const userMessage: Message = {
      id: Date.now().toString(),
      text: reply,
      isBot: false,
      timestamp: new Date()
    };

    setMessages(prev => [...prev, userMessage]);
    setIsTyping(true);

    // Bot yanıtını hazırla
    setTimeout(() => {
      let botResponse = "";
      let botQuickReplies: string[] = [];

      // Q&A eşleşmesi varsa öncelik ver
      if (qnaMatch) {
        botResponse = qnaMatch.answer;
        botQuickReplies = ["Başka soru", "Menüyü görüntüle", "Kategoriler"];
      } else {
        // Direct kategori eşleştirmesi
        const replyLower = reply.toLowerCase();
        
        if (replyLower.includes("kahvaltı") || reply === "Kahvaltılar") {
          // Kahvaltı kategorisindeki menu itemları direkt göster
          if (categories && menuItems) {
            const category = categories.find(cat => cat.slug === "kahvaltilar");
            if (category) {
              const items = menuItems.filter(item => 
                item.categoryId === category.id && item.isActive
              );
              
              if (items.length > 0) {
                const itemsList = items.map(item => `• ${item.name} - ₺${item.price}`).join('\n');
                botResponse = `🥐 **Kahvaltı Menümüz:**\n\n${itemsList}\n\nDetayları görmek için kategoriye gitmek ister misiniz?`;
                botQuickReplies = ["Kahvaltı Kategorisine Git", "Başka kategori", "Menüyü görüntüle"];
              } else {
                botResponse = "Kahvaltı kategorisinde henüz ürün bulunmamaktadır. Admin panelinden ürün eklenmesini bekleyiniz.";
                botQuickReplies = ["Başka kategori", "Menüyü görüntüle"];
              }
            } else {
              botResponse = "Kahvaltı kategorisi bulunamadı.";
              botQuickReplies = ["Menüyü görüntüle"];
            }
          } else {
            botResponse = "Menü yükleniyor, lütfen bekleyin...";
            botQuickReplies = ["Tekrar dene"];
          }
        } else if (reply === "Kahvaltı Kategorisine Git") {
          botResponse = "Sizi kahvaltı sayfasına yönlendiriyorum...";
          setTimeout(() => {
            setLocation("/category/kahvaltilar");
            onClose();
          }, 1000);
        } else if (replyLower.includes("ana yemek") || reply === "Ana Yemekler") {
          // Ana yemek kategorisindeki menu itemları direkt göster
          if (categories && menuItems) {
            const category = categories.find(cat => cat.slug === "ana-yemekler");
            if (category) {
              const items = menuItems.filter(item => 
                item.categoryId === category.id && item.isActive
              );
              
              if (items.length > 0) {
                const itemsList = items.map(item => `• ${item.name} - ₺${item.price}`).join('\n');
                botResponse = `🍽️ **Ana Yemek Menümüz:**\n\n${itemsList}\n\nDetayları görmek için kategoriye gitmek ister misiniz?`;
                botQuickReplies = ["Ana Yemek Kategorisine Git", "Başka kategori", "Menüyü görüntüle"];
              } else {
                botResponse = "Ana yemek kategorisinde henüz ürün bulunmamaktadır. Admin panelinden ürün eklenmesini bekleyiniz.";
                botQuickReplies = ["Başka kategori", "Menüyü görüntüle"];
              }
            } else {
              botResponse = "Ana yemek kategorisi bulunamadı.";
              botQuickReplies = ["Menüyü görüntüle"];
            }
          } else {
            botResponse = "Menü yükleniyor, lütfen bekleyin...";
            botQuickReplies = ["Tekrar dene"];
          }
        } else if (reply === "Ana Yemek Kategorisine Git") {
          botResponse = "Sizi ana yemek sayfasına yönlendiriyorum...";
          setTimeout(() => {
            setLocation("/category/ana-yemekler");
            onClose();
          }, 1000);
        } else if (replyLower.includes("içecek") || reply === "İçecekler") {
          // İçecek kategorisindeki menu itemları direkt göster
          if (categories && menuItems) {
            const category = categories.find(cat => cat.slug === "icecekler");
            if (category) {
              const items = menuItems.filter(item => 
                item.categoryId === category.id && item.isActive
              );
              
              if (items.length > 0) {
                const itemsList = items.map(item => `• ${item.name} - ₺${item.price}`).join('\n');
                botResponse = `☕ **İçecek Menümüz:**\n\n${itemsList}\n\nDetayları görmek için kategoriye gitmek ister misiniz?`;
                botQuickReplies = ["İçecek Kategorisine Git", "Başka kategori", "Menüyü görüntüle"];
              } else {
                botResponse = "İçecek kategorisinde henüz ürün bulunmamaktadır. Admin panelinden ürün eklenmesini bekleyiniz.";
                botQuickReplies = ["Başka kategori", "Menüyü görüntüle"];
              }
            } else {
              botResponse = "İçecek kategorisi bulunamadı.";
              botQuickReplies = ["Menüyü görüntüle"];
            }
          } else {
            botResponse = "Menü yükleniyor, lütfen bekleyin...";
            botQuickReplies = ["Tekrar dene"];
          }
        } else if (reply === "İçecek Kategorisine Git") {
          botResponse = "Sizi içecek sayfasına yönlendiriyorum...";
          setTimeout(() => {
            setLocation("/category/icecekler");
            onClose();
          }, 1000);
        } else if (replyLower.includes("tatlı") || reply === "Tatlılar") {
          // Tatlı kategorisindeki menu itemları direkt göster
          if (categories && menuItems) {
            const category = categories.find(cat => cat.slug === "tatlilar");
            if (category) {
              const items = menuItems.filter(item => 
                item.categoryId === category.id && item.isActive
              );
              
              if (items.length > 0) {
                const itemsList = items.map(item => `• ${item.name} - ₺${item.price}`).join('\n');
                botResponse = `🍰 **Tatlı Menümüz:**\n\n${itemsList}\n\nDetayları görmek için kategoriye gitmek ister misiniz?`;
                botQuickReplies = ["Tatlı Kategorisine Git", "Başka kategori", "Menüyü görüntüle"];
              } else {
                botResponse = "Tatlı kategorisinde henüz ürün bulunmamaktadır. Admin panelinden ürün eklenmesini bekleyiniz.";
                botQuickReplies = ["Başka kategori", "Menüyü görüntüle"];
              }
            } else {
              botResponse = "Tatlı kategorisi bulunamadı.";
              botQuickReplies = ["Menüyü görüntüle"];
            }
          } else {
            botResponse = "Menü yükleniyor, lütfen bekleyin...";
            botQuickReplies = ["Tekrar dene"];
          }
        } else if (reply === "Tatlı Kategorisine Git") {
          botResponse = "Sizi tatlı sayfasına yönlendiriyorum...";
          setTimeout(() => {
            setLocation("/category/tatlilar");
            onClose();
          }, 1000);
        } else if (reply === "Google'da Yorumla" || replyLower.includes("yorum") || replyLower.includes("google")) {
          botResponse = "Deneyiminizi Google'da paylaşmak istediğiniz için teşekkür ederiz! 🌟 Google yorumlarınız bizim için çok değerli. Sizi Google sayfamıza yönlendiriyorum...";
          botQuickReplies = ["5 Yıldız Ver", "Başka soru"];
          // Google yorumlar sayfasını aç
          setTimeout(() => {
            const googleUrl = "https://www.google.com/maps/search/?api=1&query=Jouska+Cafe+Restaurant";
            window.open(googleUrl, '_blank');
          }, 2000);
        } else if (replyLower.includes("menü") || reply === "Menüyü görüntüle") {
          botResponse = "Sizi ana menüye yönlendiriyorum! 🍽️ Tüm lezzetli seçeneklerimizi keşfedin.";
          setTimeout(() => {
            setLocation("/");
            onClose();
          }, 2000);
        } else if (quickReplyAction) {
          switch (quickReplyAction.action) {
            case "show_menu":
              botResponse = "Harika! Menümüzde birçok lezzetli seçenek var. Hangi kategoriye bakmak istersiniz?";
              botQuickReplies = ["Kahvaltılar", "Ana Yemekler", "İçecekler", "Tatlılar"];
              break;
            case "popular":
              onSearch("popüler");
              onClose();
              botResponse = "En popüler ürünlerimizi listeledim! Müşterilerimizin favorilerini görebilirsiniz.";
              break;
            case "search_vejetaryen":
              onSearch("vejetaryen");
              onClose();
              botResponse = "Vejetaryen seçeneklerimizi buldum! Sağlıklı ve lezzetli alternatiflerimizi inceleyebilirsiniz.";
              break;
            case "search_hızlı":
              onSearch("hızlı");
              onClose();
              botResponse = "Hızlı servis seçeneklerimizi listeledim! Kısa sürede hazırlanan lezzetlerimizi görebilirsiniz.";
              break;
            default:
              botResponse = "Size yardımcı olmak için buradayım! Başka ne aramak istersiniz?";
              botQuickReplies = ["Arama yap", "Menüyü görüntüle", "Başka kategori"];
          }
        } else if (replyLower.includes("popüler") || reply === "Popüler Ürünler") {
          onSearch("popüler");
          onClose();
          botResponse = "En popüler ürünlerimizi listeledim! Müşterilerimizin favorilerini görebilirsiniz.";
        } else if (replyLower.includes("vejetaryen")) {
          onSearch("vejetaryen");
          onClose();
          botResponse = "Vejetaryen seçeneklerimizi buldum! Sağlıklı ve lezzetli alternatiflerimizi inceleyebilirsiniz.";
        } else if (replyLower.includes("hızlı")) {
          onSearch("hızlı");
          onClose();
          botResponse = "Hızlı servis seçeneklerimizi listeledim! Kısa sürede hazırlanan lezzetlerimizi görebilirsiniz.";
        } else if (replyLower.includes("sipariş") || reply === "Sipariş Ver") {
          if (chatbotConfig?.features?.orderSystem?.enabled) {
            botResponse = `🛒 Sipariş vermek için beğendiğiniz ürünleri seçin! Minimum sipariş tutarı ${chatbotConfig.features.orderSystem.minOrderAmount}₺, teslimat ücreti ${chatbotConfig.features.orderSystem.deliveryFee}₺. Tahmini teslimat süresi: ${chatbotConfig.features.orderSystem.estimatedDeliveryTime}`;
            botQuickReplies = ["Menüyü görüntüle", "Kahvaltılar", "Ana Yemekler", "Popüler Ürünler"];
          } else {
            botResponse = "Üzgünüm, şu anda online sipariş sistemimiz aktif değil. Telefon ile sipariş verebilirsiniz.";
          }
        } else if (replyLower.includes("rezervasyon") || reply === "Rezervasyon Yap") {
          if (chatbotConfig?.features?.reservationSystem?.enabled) {
            botResponse = `📅 Rezervasyon yapmak için istediğiniz tarihi ve saati belirtin. Maksimum ${chatbotConfig.features.reservationSystem.maxPartySize} kişilik grup rezervasyonu alıyoruz. ${chatbotConfig.features.reservationSystem.maxAdvanceDays} gün öncesinden rezervasyon yapabilirsiniz.`;
            botQuickReplies = ["Bugün için rezervasyon", "Yarın için rezervasyon", "Hafta sonu rezervasyonu"];
          } else {
            botResponse = "Rezervasyon için lütfen bizi telefon ile arayın. Size en uygun masayı ayırlayalım.";
          }
        } else if (replyLower.includes("whatsapp") || reply === "WhatsApp") {
          if (chatbotConfig?.features?.socialFeatures?.whatsappEnabled && chatbotConfig.features.socialFeatures.whatsappNumber) {
            botResponse = `📱 WhatsApp üzerinden de sipariş verebilirsiniz! ${chatbotConfig.features.socialFeatures.whatsappNumber} numarasından bizimle iletişime geçin.`;
            // WhatsApp linkini aç
            setTimeout(() => {
              const whatsappUrl = `https://wa.me/${chatbotConfig.features.socialFeatures.whatsappNumber.replace(/[^0-9]/g, '')}?text=Merhaba, menünüzü inceledim ve sipariş vermek istiyorum.`;
              window.open(whatsappUrl, '_blank');
            }, 2000);
          } else {
            botResponse = "WhatsApp sipariş hattımız şu anda aktif değil. Menümüzden sipariş verebilir veya telefon ile arayabilirsiniz.";
          }
        } else {
          // Müşteri sorusunu kaydet
          if (chatbotConfig) {
            const customerQuestion = {
              id: Date.now().toString(),
              question: reply,
              timestamp: new Date().toISOString(),
              hasAnswer: false,
              frequency: 1
            };
            
            const updatedConfig = {
              ...chatbotConfig,
              customerQuestions: [...(chatbotConfig.customerQuestions || []), customerQuestion]
            };
            
            localStorage.setItem('chatbotConfig', JSON.stringify(updatedConfig));
            window.dispatchEvent(new CustomEvent('chatbotConfigUpdated'));
          }
          
          botResponse = "Size yardımcı olmak için buradayım! Başka ne aramak istersiniz?";
          botQuickReplies = ["Arama yap", "Menüyü görüntüle", "Başka kategori"];
        }
      }

      const botMessage: Message = {
        id: Date.now().toString(),
        text: botResponse,
        isBot: true,
        timestamp: new Date(),
        quickReplies: botQuickReplies.length > 0 ? botQuickReplies : ["Arama yap", "Menüyü görüntüle", "Başka kategori"]
      };

      setMessages(prev => [...prev, botMessage]);
      setIsTyping(false);
    }, 1500);
  };

  // Müşteri sorusunu kaydet
  const saveCustomerQuestion = (question: string, hasAnswer: boolean, answerGiven?: string) => {
    try {
      const savedConfig = localStorage.getItem('chatbotConfig');
      if (savedConfig) {
        const config = JSON.parse(savedConfig);
        const existingQuestion = config.customerQuestions?.find((q: any) => q.question.toLowerCase() === question.toLowerCase());
        
        if (existingQuestion) {
          // Soru daha önce sorulmuş, frekansı artır
          existingQuestion.frequency++;
          existingQuestion.timestamp = new Date().toISOString();
          if (hasAnswer && answerGiven) {
            existingQuestion.hasAnswer = true;
            existingQuestion.answerGiven = answerGiven;
          }
        } else {
          // Yeni soru ekle
          const newQuestion = {
            id: Date.now().toString(),
            question: question,
            timestamp: new Date().toISOString(),
            hasAnswer: hasAnswer,
            answerGiven: answerGiven || "",
            frequency: 1
          };
          
          if (!config.customerQuestions) {
            config.customerQuestions = [];
          }
          config.customerQuestions.push(newQuestion);
        }
        
        localStorage.setItem('chatbotConfig', JSON.stringify(config));
        console.log('Customer question saved:', question);
      }
    } catch (error) {
      console.error('Error saving customer question:', error);
    }
  };

  const handleSendMessage = () => {
    if (!inputValue.trim()) return;

    const userText = inputValue.trim();
    const qnaMatch = findQnaMatch(userText);

    const userMessage: Message = {
      id: Date.now().toString(),
      text: userText,
      isBot: false,
      timestamp: new Date()
    };

    setMessages(prev => [...prev, userMessage]);
    setIsTyping(true);

    setTimeout(() => {
      let botResponse = "";
      let botQuickReplies: string[] = [];

      if (qnaMatch) {
        // Q&A eşleşmesi bulundu
        botResponse = qnaMatch.answer;
        botQuickReplies = ["Başka soru", "Menüyü görüntüle", "Kategoriler"];
        // Soruyu cevaplandırılmış olarak kaydet
        saveCustomerQuestion(userText, true, qnaMatch.answer);
      } else {
        // Arama yap
        onSearch(userText);
        botResponse = `"${userText}" için arama sonuçlarını görüntülüyorum! Menümüzde eşleşen ürünleri bulabilirsiniz.`;
        botQuickReplies = ["Başka arama", "Kategorileri gör", "Popüler ürünler"];
        // Soruyu cevapsız olarak kaydet
        saveCustomerQuestion(userText, false);
      }

      const botMessage: Message = {
        id: Date.now().toString(),
        text: botResponse,
        isBot: true,
        timestamp: new Date(),
        quickReplies: botQuickReplies
      };

      setMessages(prev => [...prev, botMessage]);
      setIsTyping(false);
    }, 1000);

    setInputValue("");
  };

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === "Enter") {
      handleSendMessage();
    }
  };

  if (!isOpen) return null;

  return (
    <AnimatePresence>
      <motion.div
        initial={{ opacity: 0 }}
        animate={{ opacity: 1 }}
        exit={{ opacity: 0 }}
        className="fixed inset-0 bg-black/50 z-[9999] flex items-center justify-center p-2 sm:p-4 overflow-hidden"
      >
        <motion.div
          initial={{ scale: 0.9, opacity: 0 }}
          animate={{ scale: 1, opacity: 1 }}
          exit={{ scale: 0.9, opacity: 0 }}
          className="w-full max-w-md sm:max-w-lg relative"
          style={{ position: 'relative' }}
        >
          <Card className="h-[600px] sm:h-[650px] flex flex-col max-h-[95vh] sm:max-h-[90vh]">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-3">
              <div className="flex items-center space-x-2">
                <Avatar className="w-8 h-8">
                  <AvatarFallback className="bg-orange-500 text-white">
                    <MessageCircle className="w-4 h-4" />
                  </AvatarFallback>
                </Avatar>
                <div>
                  <CardTitle className="text-lg">{chatbotConfig?.botName || "Jouska Asistan"}</CardTitle>
                  <p className="text-sm text-muted-foreground">Menü Rehberiniz</p>
                </div>
              </div>
              <Button variant="ghost" size="sm" onClick={onClose}>
                <X className="w-4 h-4" />
              </Button>
            </CardHeader>

            <CardContent className="flex-1 flex flex-col p-0">
              <ScrollArea className="flex-1 px-4" style={{ maxHeight: 'calc(100% - 100px)', minHeight: '300px' }}>
                <div className="py-3 space-y-3">
                  {messages.map((message) => (
                    <div
                      key={message.id}
                      className={`flex ${message.isBot ? "justify-start" : "justify-end"} clear-both`}
                    >
                      <div
                        className={`max-w-[85%] rounded-lg px-3 py-2 break-words ${
                          message.isBot
                            ? "bg-muted text-foreground"
                            : "bg-orange-500 text-white"
                        }`}
                      >
                        <p className="text-sm">{message.text}</p>
                        {message.quickReplies && (
                          <div className="flex flex-wrap gap-1 mt-2">
                            {message.quickReplies.map((reply, index) => (
                              <Badge
                                key={index}
                                variant="secondary"
                                className="cursor-pointer hover:bg-orange-100 text-xs py-1 px-2"
                                onClick={() => handleQuickReply(reply)}
                              >
                                {reply}
                              </Badge>
                            ))}
                          </div>
                        )}
                      </div>
                    </div>
                  ))}
                  
                  {isTyping && (
                    <div className="flex justify-start clear-both">
                      <div className="bg-muted rounded-lg px-3 py-2">
                        <div className="flex space-x-1">
                          <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce"></div>
                          <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce delay-100"></div>
                          <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce delay-200"></div>
                        </div>
                      </div>
                    </div>
                  )}
                  <div ref={messagesEndRef} />
                </div>
              </ScrollArea>

              <div className="px-4 py-3 border-t bg-white">
                <div className="flex space-x-2 mb-3">
                  <Input
                    placeholder="Mesajınızı yazın..."
                    value={inputValue}
                    onChange={(e) => setInputValue(e.target.value)}
                    onKeyPress={handleKeyPress}
                    className="flex-1 h-9"
                  />
                  <Button size="sm" onClick={handleSendMessage} className="h-9 px-3">
                    <Send className="w-4 h-4" />
                  </Button>
                </div>
                
                <div className="flex flex-wrap gap-1">
                  {quickReplies.slice(0, 3).map((reply, index) => (
                    <Badge
                      key={index}
                      variant="outline"
                      className="cursor-pointer hover:bg-orange-50 text-xs flex items-center gap-1 py-1"
                      onClick={() => handleQuickReply(reply.text)}
                    >
                      {reply.icon}
                      {reply.text}
                    </Badge>
                  ))}
                </div>
              </div>
            </CardContent>
          </Card>
        </motion.div>
      </motion.div>
    </AnimatePresence>
  );
}

export default WelcomeChatbot;