import React from 'react';
import { useMenuItems } from '@/hooks/use-menu-items';
import { Category, MenuItem } from '@shared/schema';
import { useCategories } from '@/hooks/use-categories';
import { Card, CardContent } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { ChevronRight } from 'lucide-react';
import { Link } from 'wouter';

interface RelatedMenuItemsProps {
  currentCategoryId?: number;
  excludeMenuItemIds?: number[];
  maxItems?: number;
}

export function RelatedMenuItems({ 
  currentCategoryId, 
  excludeMenuItemIds = [],
  maxItems = 4
}: RelatedMenuItemsProps) {
  const { data: allMenuItems, isLoading: isLoadingMenuItems } = useMenuItems();
  const { data: categories, isLoading: isLoadingCategories } = useCategories();
  
  if (isLoadingMenuItems || isLoadingCategories || !allMenuItems || !categories) {
    return <div className="mt-8 py-4 text-center text-muted-foreground">İlgili öğeler yükleniyor...</div>;
  }
  
  // Şu anki kategoriyi bul
  const currentCategory = categories.find(c => c.id === currentCategoryId);
  
  // İlgili kategorileri bul (şu anki kategori hariç)
  const relatedCategories = getRelatedCategories(currentCategoryId, categories);
  
  if (relatedCategories.length === 0) {
    return null;
  }
  
  // İlgili kategorilerden menü öğelerini topla
  let relatedItems: (MenuItem & { category?: Category })[] = [];
  
  relatedCategories.forEach(category => {
    const categoryItems = allMenuItems
      .filter(item => 
        item.categoryId === category.id && 
        !excludeMenuItemIds.includes(item.id)
      )
      .map(item => ({
        ...item,
        category
      }));
    
    relatedItems = [...relatedItems, ...categoryItems];
  });
  
  // Rastgele karıştır ve sınırla
  relatedItems = shuffleArray(relatedItems).slice(0, maxItems);
  
  if (relatedItems.length === 0) {
    return null;
  }
  
  return (
    <div className="mt-10 pt-6 border-t">
      <div className="flex items-center justify-between mb-4">
        <h3 className="text-xl font-semibold font-display">Şunları da Sevebilirsiniz</h3>
        {currentCategory && (
          <Button variant="ghost" size="sm" asChild>
            <Link to={`/`}>
              Tüm Kategorileri Gör <ChevronRight className="ml-1 h-4 w-4" />
            </Link>
          </Button>
        )}
      </div>
      
      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
        {relatedItems.map(item => (
          <Card key={item.id} className="overflow-hidden group transition-all duration-300 hover:shadow-md">
            <div className="relative h-48 overflow-hidden bg-gray-100">
              {item.imageUrl && (
                <img
                  src={item.imageUrl}
                  alt={item.name}
                  className="w-full h-full object-cover transition-transform duration-500 group-hover:scale-105"
                />
              )}
              
              {item.category && (
                <Badge className="absolute top-2 right-2 bg-primary/80 hover:bg-primary/90">
                  {item.category.name}
                </Badge>
              )}
            </div>
            
            <CardContent className="pt-3">
              <div className="flex justify-between items-start">
                <div>
                  <h4 className="font-medium truncate">{item.name}</h4>
                  <p className="text-sm text-muted-foreground line-clamp-2 h-10">
                    {item.description}
                  </p>
                </div>
                <div className="font-semibold text-primary whitespace-nowrap">
                  {item.price}
                </div>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>
    </div>
  );
}

// İlgili kategorileri getir
function getRelatedCategories(currentCategoryId: number | undefined, categories: Category[]): Category[] {
  if (!currentCategoryId) {
    // Tüm kategorileri rastgele sırayla döndür
    return shuffleArray([...categories]).slice(0, 3);
  }
  
  // Kategori ilişkileri (gerçek bir uygulamada bu veriler veritabanından gelebilir)
  const categoryRelations: Record<string, string[]> = {
    // Ana Yemekler ilişkileri
    'ana-yemekler': ['makarnalar', 'salatalar', 'burgerler'],
    // Kahvaltı ilişkileri
    'kahvaltilar': ['omletler', 'gozlemeler', 'tostlar'],
    // İçecek ilişkileri
    'soguk-icecekler': ['sicak-icecekler'],
    'sicak-icecekler': ['soguk-icecekler'],
    // Tatlı ilişkileri
    'sutlu-tatlilar': ['pasta-ve-tartlar', 'cheesecake', 'waffle', 'dondurma', 'kremalar'],
    'cheesecake': ['pasta-ve-tartlar', 'sutlu-tatlilar', 'kremalar'],
    'pasta-ve-tartlar': ['cheesecake', 'sutlu-tatlilar', 'kremalar'],
    'waffle': ['dondurma', 'sutlu-tatlilar'],
    'dondurma': ['waffle', 'sutlu-tatlilar'],
    'kremalar': ['sutlu-tatlilar', 'pasta-ve-tartlar', 'cheesecake'],
    // Diğer ilişkiler
    'burgerler': ['ana-yemekler', 'tostlar', 'makarnalar'],
    'tostlar': ['kahvaltilar', 'burgerler'],
    'makarnalar': ['ana-yemekler', 'burgerler'],
    'salatalar': ['ana-yemekler'],
    'omletler': ['kahvaltilar', 'gozlemeler'],
    'gozlemeler': ['kahvaltilar', 'omletler', 'tostlar'],
    'atistirmaliklar': ['salatalar', 'tostlar']
  };
  
  // Şu anki kategoriyi bul
  const currentCategory = categories.find(c => c.id === currentCategoryId);
  if (!currentCategory) {
    return shuffleArray([...categories]).slice(0, 3);
  }
  
  // İlgili kategori sluglarını al
  const relatedSlugs = categoryRelations[currentCategory.slug] || [];
  
  // İlgili kategorileri bul
  const relatedCats = categories.filter(c => 
    c.id !== currentCategoryId && 
    relatedSlugs.includes(c.slug)
  );
  
  // Eğer ilgili kategori bulunamazsa, rastgele kategoriler döndür
  if (relatedCats.length === 0) {
    return shuffleArray(
      categories.filter(c => c.id !== currentCategoryId)
    ).slice(0, 3);
  }
  
  return relatedCats;
}

// Diziyi karıştır
function shuffleArray<T>(array: T[]): T[] {
  const newArray = [...array];
  for (let i = newArray.length - 1; i > 0; i--) {
    const j = Math.floor(Math.random() * (i + 1));
    [newArray[i], newArray[j]] = [newArray[j], newArray[i]];
  }
  return newArray;
}