import React from 'react';
import { AlertCircle, Wifi, WifiOff, RefreshCw, Clock } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { getCacheAge, isDataCached, isCacheStale } from '@/lib/offline-storage';

interface OfflineBannerProps {
  isOnline: boolean;
  onRefresh: () => void;
}

export function OfflineBanner({ isOnline, onRefresh }: OfflineBannerProps) {
  const dataCached = isDataCached();
  const cacheStale = isCacheStale();
  const cacheAge = getCacheAge();
  
  // Don't show the banner if we're online and cache is not stale
  if (isOnline && (!dataCached || !cacheStale)) {
    return null;
  }
  
  let status = 'unknown';
  let icon = <AlertCircle className="h-5 w-5" />;
  let color = 'bg-orange-100 border-orange-300 text-orange-800';
  let message = 'Bilinmeyen bağlantı durumu';
  
  if (!isOnline && !dataCached) {
    // Offline, no cached data
    status = 'offline-no-cache';
    icon = <WifiOff className="h-5 w-5" />;
    color = 'bg-red-100 border-red-300 text-red-800';
    message = 'Bağlantı yok ve çevrimdışı veri bulunamadı. Lütfen internet bağlantınızı kontrol ediniz.';
  } else if (!isOnline && dataCached) {
    // Offline, using cached data
    status = 'offline-with-cache';
    icon = <Clock className="h-5 w-5" />;
    color = 'bg-amber-100 border-amber-300 text-amber-800';
    message = `Çevrimdışı moddasınız. Kaydedilmiş menü verilerini görüntülüyorsunuz (${cacheAge} dakika önce kaydedildi).`;
  } else if (isOnline && dataCached && cacheStale) {
    // Online, but cache is stale
    status = 'online-stale-cache';
    icon = <Wifi className="h-5 w-5" />;
    color = 'bg-blue-100 border-blue-300 text-blue-800';
    message = 'Menü verileri güncel olmayabilir. Yenilemek için tıklayınız.';
  }
  
  return (
    <Alert className={`${color} mb-4`}>
      <div className="flex items-center justify-between">
        <div className="flex items-center">
          {icon}
          <AlertDescription className="ml-2">
            {message}
          </AlertDescription>
        </div>
        
        {isOnline && (
          <Button 
            size="sm" 
            variant="outline" 
            className="border-current text-current hover:bg-current hover:bg-opacity-10 hover:text-current"
            onClick={onRefresh}
          >
            <RefreshCw className="h-4 w-4 mr-1" />
            Yenile
          </Button>
        )}
      </div>
    </Alert>
  );
}