import React, { useState, useEffect } from "react";
import { MenuItem } from "@shared/schema";
import { MenuItemDetailModal } from "./menu-item-detail-modal";
import { Heart, Plus, ShoppingCart } from "lucide-react";
import { Button } from "@/components/ui/button";
import { useQuery } from "@tanstack/react-query";

interface MenuItemProps {
  item: MenuItem;
}

export const MenuItemCard = ({ item }: MenuItemProps) => {
  const [detailModalOpen, setDetailModalOpen] = useState(false);
  const [isFavorite, setIsFavorite] = useState(false);
  
  // Restaurant info'yu çekip sipariş sistemi durumunu kontrol ediyoruz
  const { data: restaurantInfo } = useQuery({
    queryKey: ['/api/restaurant'],
  });
  
  // Debug: Log item data
  useEffect(() => {
    console.log('MenuItemCard item:', {
      id: item.id,
      name: item.name,
      image: item.image,
      hasImage: !!item.image
    });
  }, [item]);
  
  // Load favorites from localStorage on component mount
  useEffect(() => {
    if (typeof window !== 'undefined') {
      const favorites = JSON.parse(localStorage.getItem('favorites') || '[]');
      setIsFavorite(favorites.some((fav: number) => fav === item.id));
    }
  }, [item.id]);
  
  // Handle favorite toggling
  const toggleFavorite = (e: React.MouseEvent) => {
    e.stopPropagation(); // Prevent modal from opening
    
    if (typeof window !== 'undefined') {
      const favorites = JSON.parse(localStorage.getItem('favorites') || '[]');
      
      if (isFavorite) {
        // Remove from favorites
        const updatedFavorites = favorites.filter((id: number) => id !== item.id);
        localStorage.setItem('favorites', JSON.stringify(updatedFavorites));
        setIsFavorite(false);
        
        // Dispatch custom event to notify other components
        window.dispatchEvent(new Event('favoritesUpdated'));
      } else {
        // Add to favorites
        const updatedFavorites = [...favorites, item.id];
        localStorage.setItem('favorites', JSON.stringify(updatedFavorites));
        setIsFavorite(true);
        
        // Dispatch custom event to notify other components
        window.dispatchEvent(new Event('favoritesUpdated'));
      }
    }
  };


  
  return (
    <>
      <div 
        className="menu-item-card relative bg-white rounded-lg overflow-hidden transition-all duration-300 shadow-sm hover:shadow-md cursor-pointer group border border-gray-100 flex flex-col sm:flex-row"
        onClick={() => setDetailModalOpen(true)}
      >
        {/* Favorite button */}
        <button 
          onClick={toggleFavorite}
          className={`absolute top-2 right-2 z-30 p-1.5 rounded-full shadow-sm transition-colors duration-300 ${
            isFavorite 
              ? 'bg-red-500 text-white hover:bg-red-600' 
              : 'bg-white/90 text-gray-500 hover:bg-white'
          }`}
          aria-label={isFavorite ? "Favorilerden çıkar" : "Favorilere ekle"}
        >
          <Heart className={`h-3.5 w-3.5 ${isFavorite ? 'fill-current' : ''}`} />
        </button>
        
        {/* Image section */}
        {item.image ? (
          <div className="relative w-full sm:w-1/3 sm:min-w-[150px] h-[120px] sm:h-[140px] overflow-hidden rounded-t-lg sm:rounded-l-lg sm:rounded-t-none bg-gray-100 flex-shrink-0">
            <img 
              src={item.image} 
              alt={item.name} 
              className="w-full h-full object-cover object-center block"
              loading="eager"
              decoding="async"
              onLoad={() => console.log(`Image loaded: ${item.name}`)}
              onError={(e) => {
                console.log(`Image error for ${item.name}:`, e);
                e.currentTarget.style.display = 'none';
                e.currentTarget.parentElement!.innerHTML = `<div class="w-full h-full bg-gray-200 flex items-center justify-center"><span class="text-gray-500 text-sm">Görsel yüklenemedi</span></div>`;
              }}
            />
          </div>
        ) : (
          <div className="relative w-full sm:w-1/3 sm:min-w-[150px] h-[120px] sm:h-[140px] bg-gray-200 rounded-t-lg sm:rounded-l-lg sm:rounded-t-none flex items-center justify-center flex-shrink-0">
            <span className="text-gray-500 text-sm">Görsel yok</span>
          </div>
        )}
        
        {/* Content section */}
        <div className={`flex-1 p-4 sm:p-5 flex flex-col justify-between ${!item.image ? 'ml-2' : ''}`}>
          <div>
            <div className="flex justify-between items-start mb-3">
              <h3 className="font-display font-bold text-xl text-gray-800 leading-tight">
                {item.name}
              </h3>
            </div>
            
            {item.description && (
              <p className="text-base text-gray-600 line-clamp-2 mb-3">{item.description}</p>
            )}
          </div>
          
          {/* Tags */}
          <div className="flex gap-2 mb-3">
            {item.isVegetarian && (
              <span className="bg-green-50 text-green-700 text-sm px-3 py-1.5 rounded-full font-medium">
                Vejetaryen
              </span>
            )}
            {item.isSpicy && (
              <span className="bg-red-50 text-red-700 text-sm px-3 py-1.5 rounded-full font-medium">
                Acı
              </span>
            )}
          </div>
          
          {/* Price and Order Button */}
          <div className="flex justify-between items-center mt-3">
            <div className="font-bold text-primary text-xl">
              ₺{item.price.replace(' ₺', '')}
            </div>
            
            {/* Sipariş butonu sadece sipariş sistemi aktifse görünür */}
            {restaurantInfo?.enableOrderSystem && (
              <Button
                onClick={(e) => {
                  e.stopPropagation();
                  // Buraya sepete ekleme veya sipariş işlemi gelecek
                  alert(`${item.name} sipariş verildi!`);
                }}
                size="sm"
                className="bg-orange-500 hover:bg-orange-600 text-white px-3 py-2 rounded-lg flex items-center gap-2 shadow-sm transition-all duration-200 hover:shadow-md"
              >
                <ShoppingCart className="w-4 h-4" />
                Sipariş Ver
              </Button>
            )}
          </div>
        </div>
        
        {/* Arrow indicator */}
        <div className="absolute right-2 bottom-2 text-primary/70">
          <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4" viewBox="0 0 20 20" fill="currentColor">
            <path fillRule="evenodd" d="M7.293 14.707a1 1 0 010-1.414L10.586 10 7.293 6.707a1 1 0 011.414-1.414l4 4a1 1 0 010 1.414l-4 4a1 1 0 01-1.414 0z" clipRule="evenodd" />
          </svg>
        </div>
      </div>
      
      {detailModalOpen && (
        <MenuItemDetailModal 
          open={detailModalOpen}
          onOpenChange={setDetailModalOpen}
          item={item}
        />
      )}
    </>
  );
};
