import React from "react";
import { Button } from "@/components/ui/button";
import { QrCode, Settings, Search, X } from "lucide-react";
import { Restaurant } from "@shared/schema";
import { Link } from "wouter";
import { ThemeSwitcher } from "../theme-switcher";
import { Input } from "@/components/ui/input";

interface HeaderProps {
  restaurantInfo?: Restaurant;
  onQrButtonClick: () => void;
  searchQuery: string;
  setSearchQuery: (query: string) => void;
}

export const Header = ({ restaurantInfo, onQrButtonClick, searchQuery, setSearchQuery }: HeaderProps) => {
  // Check if the current URL contains a query parameter for admin mode
  const isAdminMode = typeof window !== 'undefined' && window.location.search.includes('admin=true');
  
  return (
    <header className="bg-primary text-white shadow-md">
      <div className="container mx-auto px-4 py-4 flex justify-between items-center">
        <div className="flex items-center">
          {/* Logo Image */}
          <div className="mr-3">
            <img 
              src="/images/logo-jouska.png" 
              alt="Jouska Logo" 
              className="h-10 md:h-12"
            />
          </div>
          
          <div className="hidden sm:block">
            <h1 className="font-display text-2xl font-bold text-white">
              {restaurantInfo?.name || "Jouska"}
            </h1>
          </div>
        </div>
        
        <div className="flex items-center gap-3">
          {/* Search input */}
          <div className="relative hidden md:flex items-center">
            <div className="relative">
              <Search className="absolute left-2.5 top-2.5 h-4 w-4 text-white/70" />
              <Input
                type="search"
                placeholder="Menüde ara..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="w-44 lg:w-64 h-9 pl-9 bg-white/10 border-white/20 text-white placeholder:text-white/60 focus-visible:ring-accent"
              />
              {searchQuery && (
                <button
                  type="button"
                  onClick={() => setSearchQuery('')}
                  className="absolute right-2.5 top-2.5 text-white/70 hover:text-white"
                >
                  <X className="h-4 w-4" />
                </button>
              )}
            </div>
            {searchQuery && (
              <div className="absolute top-full mt-1 right-0 z-50 bg-popover border border-border rounded-md shadow-lg w-44 lg:w-64 py-1 text-sm">
                <div className="flex flex-wrap gap-1 p-1.5">
                  {["kahvaltı", "burger", "tatlı", "içecek"].map((keyword) => (
                    <button
                      key={keyword}
                      onClick={() => setSearchQuery(keyword)}
                      className="px-2 py-0.5 bg-primary/10 hover:bg-primary/20 rounded-full text-xs text-primary transition-colors"
                    >
                      {keyword}
                    </button>
                  ))}
                </div>
              </div>
            )}
          </div>
          
          <ThemeSwitcher />
          
          <Button 
            onClick={onQrButtonClick} 
            className="bg-accent text-white px-4 py-2 rounded-md text-sm font-medium flex items-center hover:bg-accent/90 transition-all duration-200 shadow-md"
          >
            <QrCode className="mr-2 h-4 w-4" />
            <span className="hidden sm:inline">QR Kodu</span>
          </Button>
          
          {/* Mobile search button */}
          <Button 
            variant="outline" 
            size="icon"
            className="md:hidden bg-white/10 text-white border-white/20 rounded-md text-sm hover:bg-white/20 transition-all duration-200"
            onClick={() => {
              if (typeof document !== 'undefined') {
                const searchInput = document.getElementById('mobile-search-input');
                if (searchInput) {
                  searchInput.focus();
                }
              }
            }}
          >
            <Search className="h-4 w-4" />
          </Button>
          
          {/* Only show admin button if admin mode is active */}
          {isAdminMode && (
            <Link href="/yonetim-paneli" className="no-print hidden sm:block">
              <Button 
                variant="outline"
                className="bg-white/10 text-white border-white/20 px-4 py-2 rounded-md text-sm font-medium flex items-center hover:bg-white/20 transition-all duration-200"
              >
                <Settings className="mr-2 h-4 w-4" />
                <span>Yönetim</span>
              </Button>
            </Link>
          )}
        </div>
      </div>
    </header>
  );
};
