import React, { useState } from "react";
import { Restaurant } from "@shared/schema";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { useQueryClient } from "@tanstack/react-query";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { Loader2, Store } from "lucide-react";

interface RestaurantSettingsProps {
  restaurantInfo?: Restaurant;
}

export const RestaurantSettings = ({ restaurantInfo }: RestaurantSettingsProps) => {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [formData, setFormData] = useState({
    name: restaurantInfo?.name || "",
    address: restaurantInfo?.address || "",
    phone: restaurantInfo?.phone || "",
    email: restaurantInfo?.email || "",
    website: restaurantInfo?.website || "",
    description: restaurantInfo?.description || "",
  });

  // Update form data
  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: value
    }));
  };

  // Update restaurant info
  const handleUpdateRestaurantInfo = async () => {
    setIsSubmitting(true);
    try {
      const response = await apiRequest("PATCH", "/api/restaurant", formData);
      
      if (response.ok) {
        const updatedInfo = await response.json();
        queryClient.setQueryData(["/api/restaurant"], updatedInfo);
        
        toast({
          title: "Başarılı",
          description: "Restoran bilgileri güncellendi",
        });
      } else {
        throw new Error("Güncelleme başarısız");
      }
    } catch (error) {
      console.error("Restaurant update error:", error);
      toast({
        title: "Hata",
        description: "Restoran bilgileri güncellenirken bir hata oluştu",
        variant: "destructive",
      });
    } finally {
      setIsSubmitting(false);
    }
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center gap-2 mb-6">
        <Store className="w-6 h-6 text-primary" />
        <h2 className="text-2xl font-bold text-gray-900">Restoran Bilgileri</h2>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Temel Bilgiler</CardTitle>
          <CardDescription>
            Restoranınızın temel bilgilerini buradan düzenleyebilirsiniz
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div className="space-y-2">
              <label htmlFor="name" className="text-sm font-medium">Restoran Adı *</label>
              <Input
                id="name"
                name="name"
                value={formData.name}
                onChange={handleInputChange}
                placeholder="Örn: Jouska Cafe & Restaurant"
                required
              />
            </div>
            
            <div className="space-y-2">
              <label htmlFor="phone" className="text-sm font-medium">Telefon</label>
              <Input
                id="phone"
                name="phone"
                value={formData.phone}
                onChange={handleInputChange}
                placeholder="Örn: (0212) 123 45 67"
              />
            </div>
            
            <div className="space-y-2">
              <label htmlFor="email" className="text-sm font-medium">E-posta</label>
              <Input
                id="email"
                name="email"
                type="email"
                value={formData.email}
                onChange={handleInputChange}
                placeholder="Örn: info@jouska.com"
              />
            </div>
            
            <div className="space-y-2">
              <label htmlFor="website" className="text-sm font-medium">Website</label>
              <Input
                id="website"
                name="website"
                value={formData.website}
                onChange={handleInputChange}
                placeholder="Örn: https://www.jouska.com"
              />
            </div>
            
            <div className="space-y-2 md:col-span-2">
              <label htmlFor="address" className="text-sm font-medium">Adres</label>
              <Textarea
                id="address"
                name="address"
                value={formData.address}
                onChange={handleInputChange}
                placeholder="Restoranınızın tam adresi"
                rows={3}
              />
            </div>
            
            <div className="space-y-2 md:col-span-2">
              <label htmlFor="description" className="text-sm font-medium">Açıklama</label>
              <Textarea
                id="description"
                name="description"
                value={formData.description}
                onChange={handleInputChange}
                placeholder="Restoranınız hakkında kısa bir açıklama"
                rows={4}
              />
            </div>
            
            <div className="pt-4 md:col-span-2">
              <Button 
                onClick={handleUpdateRestaurantInfo}
                disabled={isSubmitting}
                className="w-full sm:w-auto"
              >
                {isSubmitting && (
                  <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                )}
                Bilgileri Kaydet
              </Button>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
};