import React, { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { 
  ShoppingCart, 
  Clock, 
  CheckCircle, 
  XCircle, 
  Eye, 
  Search,
  Filter,
  Calendar
} from "lucide-react";

interface Order {
  id: number;
  orderNumber: string;
  customerName: string;
  customerPhone?: string;
  totalAmount: string;
  status: string;
  tableNumber?: string;
  notes?: string;
  createdAt: string;
  items?: OrderItem[];
}

interface OrderItem {
  id: number;
  orderId: number;
  menuItemId: number;
  menuItemName: string;
  quantity: number;
  price: string;
  notes?: string;
}

const statusColors = {
  pending: "bg-yellow-100 text-yellow-800",
  preparing: "bg-blue-100 text-blue-800", 
  ready: "bg-green-100 text-green-800",
  delivered: "bg-gray-100 text-gray-800",
  cancelled: "bg-red-100 text-red-800"
};

const statusLabels = {
  pending: "Bekliyor",
  preparing: "Hazırlanıyor",
  ready: "Hazır",
  delivered: "Teslim Edildi",
  cancelled: "İptal Edildi"
};

export const OrderManagement = () => {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [selectedOrder, setSelectedOrder] = useState<Order | null>(null);
  const [statusFilter, setStatusFilter] = useState("all");
  const [searchTerm, setSearchTerm] = useState("");
  const [activeTab, setActiveTab] = useState("all");

  // Fetch orders
  const { data: orders = [], isLoading } = useQuery<Order[]>({
    queryKey: ['/api/orders'],
  });

  // Update order status mutation
  const updateOrderMutation = useMutation({
    mutationFn: async ({ orderId, status }: { orderId: number; status: string }) => {
      const response = await apiRequest("PATCH", `/api/orders/${orderId}/status`, { status });
      if (!response.ok) throw new Error("Sipariş durumu güncellenemedi");
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/orders'] });
      toast({
        title: "Başarılı",
        description: "Sipariş durumu güncellendi",
      });
    },
    onError: (error) => {
      toast({
        title: "Hata",
        description: "Sipariş durumu güncellenirken bir hata oluştu",
        variant: "destructive",
      });
    },
  });

  // Filter orders
  const filteredOrders = orders.filter(order => {
    const matchesSearch = 
      order.orderNumber.toLowerCase().includes(searchTerm.toLowerCase()) ||
      order.customerName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      (order.customerPhone && order.customerPhone.includes(searchTerm));
    
    const matchesStatus = statusFilter === "all" || order.status === statusFilter;
    const matchesTab = activeTab === "all" || order.status === activeTab;
    
    return matchesSearch && matchesStatus && matchesTab;
  });

  // Get order counts by status
  const getOrderCount = (status: string) => {
    if (status === "all") return orders.length;
    return orders.filter(order => order.status === status).length;
  };

  const handleStatusUpdate = (orderId: number, newStatus: string) => {
    updateOrderMutation.mutate({ orderId, status: newStatus });
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'pending': return <Clock className="w-4 h-4" />;
      case 'preparing': return <ShoppingCart className="w-4 h-4" />;
      case 'ready': return <CheckCircle className="w-4 h-4" />;
      case 'delivered': return <CheckCircle className="w-4 h-4" />;
      case 'cancelled': return <XCircle className="w-4 h-4" />;
      default: return <Clock className="w-4 h-4" />;
    }
  };

  if (isLoading) {
    return (
      <div className="space-y-4">
        <div className="flex items-center gap-2 mb-6">
          <ShoppingCart className="w-6 h-6 text-primary" />
          <h2 className="text-2xl font-bold text-gray-900">Sipariş Yönetimi</h2>
        </div>
        <div className="text-center py-8">Siparişler yükleniyor...</div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="flex items-center gap-2 mb-6">
        <ShoppingCart className="w-6 h-6 text-primary" />
        <h2 className="text-2xl font-bold text-gray-900">Sipariş Yönetimi</h2>
      </div>

      {/* Filters */}
      <Card>
        <CardHeader>
          <CardTitle>Filtreler</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex flex-col sm:flex-row gap-4">
            <div className="flex-1">
              <div className="relative">
                <Search className="absolute left-3 top-3 h-4 w-4 text-gray-400" />
                <Input
                  placeholder="Sipariş numarası, müşteri adı veya telefon ile ara..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="pl-10"
                />
              </div>
            </div>
            <Select value={statusFilter} onValueChange={setStatusFilter}>
              <SelectTrigger className="w-48">
                <Filter className="w-4 h-4 mr-2" />
                <SelectValue placeholder="Durum filtresi" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">Tüm Durumlar</SelectItem>
                <SelectItem value="pending">Bekliyor</SelectItem>
                <SelectItem value="preparing">Hazırlanıyor</SelectItem>
                <SelectItem value="ready">Hazır</SelectItem>
                <SelectItem value="delivered">Teslim Edildi</SelectItem>
                <SelectItem value="cancelled">İptal Edildi</SelectItem>
              </SelectContent>
            </Select>
          </div>
        </CardContent>
      </Card>

      {/* Order Tabs */}
      <Tabs value={activeTab} onValueChange={setActiveTab}>
        <TabsList className="grid w-full grid-cols-5">
          <TabsTrigger value="all">
            Tümü ({getOrderCount("all")})
          </TabsTrigger>
          <TabsTrigger value="pending">
            Bekliyor ({getOrderCount("pending")})
          </TabsTrigger>
          <TabsTrigger value="preparing">
            Hazırlanıyor ({getOrderCount("preparing")})
          </TabsTrigger>
          <TabsTrigger value="ready">
            Hazır ({getOrderCount("ready")})
          </TabsTrigger>
          <TabsTrigger value="delivered">
            Teslim ({getOrderCount("delivered")})
          </TabsTrigger>
        </TabsList>

        <TabsContent value={activeTab} className="mt-6">
          {filteredOrders.length === 0 ? (
            <Card>
              <CardContent className="py-8 text-center">
                <ShoppingCart className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                <p className="text-gray-500">Henüz sipariş bulunmuyor</p>
              </CardContent>
            </Card>
          ) : (
            <div className="grid gap-4">
              {filteredOrders.map((order) => (
                <Card key={order.id} className="hover:shadow-md transition-shadow">
                  <CardContent className="p-6">
                    <div className="flex flex-col lg:flex-row lg:items-center justify-between gap-4">
                      <div className="flex-1">
                        <div className="flex items-center gap-3 mb-2">
                          <h3 className="font-semibold text-lg">#{order.orderNumber}</h3>
                          <Badge className={statusColors[order.status as keyof typeof statusColors]}>
                            <span className="flex items-center gap-1">
                              {getStatusIcon(order.status)}
                              {statusLabels[order.status as keyof typeof statusLabels]}
                            </span>
                          </Badge>
                        </div>
                        
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 text-sm text-gray-600">
                          <div>
                            <span className="font-medium">Müşteri:</span> {order.customerName}
                          </div>
                          {order.customerPhone && (
                            <div>
                              <span className="font-medium">Telefon:</span> {order.customerPhone}
                            </div>
                          )}
                          {order.tableNumber && (
                            <div>
                              <span className="font-medium">Masa:</span> {order.tableNumber}
                            </div>
                          )}
                          <div>
                            <span className="font-medium">Toplam:</span> ₺{order.totalAmount}
                          </div>
                        </div>
                        
                        {order.notes && (
                          <div className="mt-2 text-sm text-gray-600">
                            <span className="font-medium">Not:</span> {order.notes}
                          </div>
                        )}
                        
                        <div className="mt-2 text-xs text-gray-500">
                          <Calendar className="w-3 h-3 inline mr-1" />
                          {new Date(order.createdAt).toLocaleString('tr-TR')}
                        </div>
                      </div>

                      <div className="flex flex-col gap-2">
                        {order.status === 'pending' && (
                          <Button
                            size="sm"
                            onClick={() => handleStatusUpdate(order.id, 'preparing')}
                            disabled={updateOrderMutation.isPending}
                          >
                            Hazırlanmaya Başla
                          </Button>
                        )}
                        
                        {order.status === 'preparing' && (
                          <Button
                            size="sm"
                            onClick={() => handleStatusUpdate(order.id, 'ready')}
                            disabled={updateOrderMutation.isPending}
                          >
                            Hazır Olarak İşaretle
                          </Button>
                        )}
                        
                        {order.status === 'ready' && (
                          <Button
                            size="sm"
                            onClick={() => handleStatusUpdate(order.id, 'delivered')}
                            disabled={updateOrderMutation.isPending}
                          >
                            Teslim Edildi
                          </Button>
                        )}
                        
                        {(order.status === 'pending' || order.status === 'preparing') && (
                          <Button
                            size="sm"
                            variant="destructive"
                            onClick={() => handleStatusUpdate(order.id, 'cancelled')}
                            disabled={updateOrderMutation.isPending}
                          >
                            İptal Et
                          </Button>
                        )}
                        
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => setSelectedOrder(order)}
                        >
                          <Eye className="w-4 h-4 mr-1" />
                          Detay
                        </Button>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          )}
        </TabsContent>
      </Tabs>

      {/* Order Detail Modal */}
      {selectedOrder && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4 z-50">
          <Card className="max-w-2xl w-full max-h-[80vh] overflow-y-auto">
            <CardHeader>
              <CardTitle>Sipariş Detayı - #{selectedOrder.orderNumber}</CardTitle>
              <Button
                variant="outline"
                size="sm"
                className="absolute top-4 right-4"
                onClick={() => setSelectedOrder(null)}
              >
                Kapat
              </Button>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-2 gap-4 text-sm">
                <div><strong>Müşteri:</strong> {selectedOrder.customerName}</div>
                <div><strong>Telefon:</strong> {selectedOrder.customerPhone || 'Belirtilmemiş'}</div>
                <div><strong>Masa:</strong> {selectedOrder.tableNumber || 'Belirtilmemiş'}</div>
                <div><strong>Toplam:</strong> ₺{selectedOrder.totalAmount}</div>
                <div><strong>Durum:</strong> 
                  <Badge className={`ml-2 ${statusColors[selectedOrder.status as keyof typeof statusColors]}`}>
                    {statusLabels[selectedOrder.status as keyof typeof statusLabels]}
                  </Badge>
                </div>
                <div><strong>Tarih:</strong> {new Date(selectedOrder.createdAt).toLocaleString('tr-TR')}</div>
              </div>
              
              {selectedOrder.notes && (
                <div>
                  <strong>Notlar:</strong>
                  <p className="mt-1 text-sm text-gray-600">{selectedOrder.notes}</p>
                </div>
              )}
              
              {selectedOrder.items && selectedOrder.items.length > 0 && (
                <div>
                  <strong>Sipariş Öğeleri:</strong>
                  <div className="mt-2 space-y-2">
                    {selectedOrder.items.map((item) => (
                      <div key={item.id} className="flex justify-between items-center py-2 border-b">
                        <div>
                          <span className="font-medium">{item.menuItemName}</span>
                          {item.notes && <p className="text-xs text-gray-500">{item.notes}</p>}
                        </div>
                        <div className="text-right">
                          <span className="text-sm">{item.quantity}x ₺{item.price}</span>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              )}
            </CardContent>
          </Card>
        </div>
      )}
    </div>
  );
};