import React, { useState } from "react";
import { MenuItem, Category } from "@shared/schema";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Dialog, DialogContent, DialogFooter, DialogHeader, DialogTitle, DialogClose } from "@/components/ui/dialog";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Switch } from "@/components/ui/switch";
import { Label } from "@/components/ui/label";
import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle } from "@/components/ui/alert-dialog";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { Edit, Image, Plus, Trash2, X, ArrowUpDown, Percent } from "lucide-react";
import { useQueryClient } from "@tanstack/react-query";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { Slider } from "@/components/ui/slider";

interface MenuItemManagementProps {
  menuItems: MenuItem[];
  categories: Category[];
}

export const MenuItemManagement = ({ menuItems, categories }: MenuItemManagementProps) => {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false);
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false);
  const [isDeleteDialogOpen, setIsDeleteDialogOpen] = useState(false);
  const [isBulkPriceUpdateOpen, setIsBulkPriceUpdateOpen] = useState(false);
  const [selectedMenuItem, setSelectedMenuItem] = useState<MenuItem | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [selectedCategory, setSelectedCategory] = useState<number | "all">("all");
  const [bulkUpdateCategory, setBulkUpdateCategory] = useState<number | "all">("all");
  const [bulkUpdatePercentage, setBulkUpdatePercentage] = useState<number>(10);
  const [bulkUpdateOperation, setBulkUpdateOperation] = useState<"increase" | "decrease">("increase");
  const [formData, setFormData] = useState({
    name: "",
    description: "",
    price: "",
    categoryId: 0,
    imageUrl: "",
    additionalImages: "",
    isAvailable: true,
    isVegetarian: false,
    isSpicy: false,
    isChefSpecial: false,
    isMonthlySpecial: false,
    isCampaign: false,
    campaignPrice: "",
    order: 0,
    preparationTime: "",
    allergens: ""
  });
  
  // Dosya yükleme için referanslar
  const fileInputRef = React.useRef<HTMLInputElement>(null);

  // Open add dialog
  const handleOpenAddDialog = () => {
    setFormData({
      name: "",
      description: "",
      price: "",
      categoryId: categories.length > 0 ? categories[0].id : 0,
      imageUrl: "",
      additionalImages: "",
      isAvailable: true,
      isVegetarian: false,
      isSpicy: false,
      isChefSpecial: false,
      isMonthlySpecial: false,
      isCampaign: false,
      campaignPrice: "",
      order: menuItems.length + 1,
      preparationTime: "",
      allergens: ""
    });
    setIsAddDialogOpen(true);
  };

  // Open edit dialog
  const handleOpenEditDialog = (menuItem: MenuItem) => {
    setSelectedMenuItem(menuItem);
    setFormData({
      name: menuItem.name,
      description: menuItem.description || "",
      price: menuItem.price,
      categoryId: menuItem.categoryId,
      imageUrl: menuItem.image || "",  // Map image to imageUrl for form
      additionalImages: menuItem.images || "",
      isAvailable: menuItem.isActive !== false,  // Map isActive to isAvailable
      isVegetarian: menuItem.isVegetarian || false,
      isSpicy: menuItem.isSpicy || false,
      isChefSpecial: menuItem.isChefSpecial || false,
      isMonthlySpecial: menuItem.isMonthlySpecial || false,
      isCampaign: menuItem.isCampaign || false,
      campaignPrice: menuItem.campaignPrice || "",
      order: menuItem.sortOrder || 0,  // Map sortOrder to order
      preparationTime: menuItem.preparationTime || "",
      allergens: menuItem.allergens || ""
    });
    setIsEditDialogOpen(true);
  };

  // Open delete dialog
  const handleOpenDeleteDialog = (menuItem: MenuItem) => {
    setSelectedMenuItem(menuItem);
    setIsDeleteDialogOpen(true);
  };

  // Update form data
  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: name === "order" || name === "categoryId" ? parseInt(value) || 0 : value
    }));
  };
  
  // Görsel önizleme
  const handleImagePreview = () => {
    if (formData.imageUrl) {
      // Görsel URL'sini doğrudan atamak yerine, yükleme durumunu kontrol edelim
      const img = document.createElement('img');
      img.onload = () => {
        // Görsel yüklendiyse, önizleme alanındaki görseli güncelle
        const imgElements = document.querySelectorAll('img[alt="Ana görsel önizleme"]');
        imgElements.forEach(imgElem => {
          (imgElem as HTMLImageElement).src = formData.imageUrl || '';
        });
      };
      img.onerror = () => {
        // Yükleme hatası durumunda yedek görsel göster ve hata bildir
        const imgElements = document.querySelectorAll('img[alt="Ana görsel önizleme"]');
        imgElements.forEach(imgElem => {
          (imgElem as HTMLImageElement).src = 'https://via.placeholder.com/200x120?text=Görsel+Yüklenemedi';
        });
        
        toast({
          title: "Görsel yüklenemedi",
          description: "Girdiğiniz URL'deki görsel yüklenemedi. Lütfen URL'yi kontrol edin.",
          variant: "destructive"
        });
      };
      img.src = formData.imageUrl;
    }
  };
  
  // Dosya yükleme işleyicisi
  const handleFileUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;
    
    // Sadece resim dosyalarını kabul et
    if (!file.type.startsWith('image/')) {
      toast({
        title: "Geçersiz dosya tipi",
        description: "Lütfen bir görsel dosyası seçin (jpeg, png, gif, vb.)",
        variant: "destructive"
      });
      return;
    }
    
    // Dosyayı base64'e çevirerek önizle
    const reader = new FileReader();
    reader.onload = (event) => {
      if (event.target?.result) {
        // Base64 kodlu görsel verisini formData'ya ekle
        const base64Image = event.target.result as string;
        setFormData(prev => ({
          ...prev,
          imageUrl: base64Image
        }));
        
        // Önizleme görseli güncelle
        const imgElements = document.querySelectorAll('img[alt="Ana görsel önizleme"]');
        imgElements.forEach(imgElem => {
          (imgElem as HTMLImageElement).src = base64Image;
        });
      }
    };
    reader.readAsDataURL(file);
  };

  // Handle select change
  const handleSelectChange = (name: string, value: string) => {
    setFormData(prev => ({
      ...prev,
      [name]: name === "categoryId" ? parseInt(value) : value
    }));
  };

  // Handle switch change
  const handleSwitchChange = (nameOrChecked: string | boolean, checkedValue?: boolean) => {
    // Eski ve yeni kullanımı destekle
    if (typeof nameOrChecked === 'boolean') {
      // Eski format: handleSwitchChange(checked)
      setFormData(prev => ({
        ...prev,
        isAvailable: nameOrChecked
      }));
    } else {
      // Yeni format: handleSwitchChange(name, checked)
      setFormData(prev => ({
        ...prev,
        [nameOrChecked]: checkedValue
      }));
    }
  };

  // Add new menu item
  const handleAddMenuItem = async () => {
    try {
      console.log("Form data isAvailable:", formData.isAvailable);
      const menuItemData = {
        name: formData.name,
        description: formData.description,
        price: formData.price,
        categoryId: formData.categoryId,
        image: formData.imageUrl,  // Map imageUrl to image for API
        images: formData.additionalImages,
        isActive: formData.isAvailable,  // Map isAvailable to isActive for API
        isVegetarian: formData.isVegetarian,
        isSpicy: formData.isSpicy,
        isChefSpecial: formData.isChefSpecial,
        isMonthlySpecial: formData.isMonthlySpecial,
        isCampaign: formData.isCampaign,
        campaignPrice: formData.campaignPrice,
        sortOrder: formData.order,  // Map order to sortOrder for API
        preparationTime: formData.preparationTime,
        allergens: formData.allergens
      };
      
      await apiRequest("/api/menu-items", {
        method: "POST",
        body: JSON.stringify(menuItemData)
      });

      queryClient.invalidateQueries({ queryKey: ['/api/menu-items'] });
      queryClient.invalidateQueries({ queryKey: ['/api/menu'] });
      toast({
        title: "Menü öğesi eklendi",
        description: `${formData.name} başarıyla eklendi.`
      });
      
      // Form'u resetle
      setFormData({
        name: "",
        description: "",
        price: "",
        categoryId: categories.length > 0 ? categories[0].id : 0,
        imageUrl: "",
        additionalImages: "",
        isAvailable: true,
        isVegetarian: false,
        isSpicy: false,
        isChefSpecial: false,
        isMonthlySpecial: false,
        isCampaign: false,
        campaignPrice: "",
        order: menuItems.length + 1,
        preparationTime: "",
        allergens: ""
      });
      
      setIsAddDialogOpen(false);
    } catch (error) {
      toast({
        title: "Hata",
        description: "Menü öğesi eklenirken bir hata oluştu.",
        variant: "destructive"
      });
    }
  };

  // Update menu item
  const handleUpdateMenuItem = async () => {
    if (!selectedMenuItem) return;
    
    try {
      const menuItemData = {
        name: formData.name,
        description: formData.description,
        price: formData.price,
        categoryId: formData.categoryId,
        image: formData.imageUrl,  // Map imageUrl to image for API
        images: formData.additionalImages,
        isActive: formData.isAvailable,  // Map isAvailable to isActive for API
        isVegetarian: formData.isVegetarian,
        isSpicy: formData.isSpicy,
        isChefSpecial: formData.isChefSpecial,
        isMonthlySpecial: formData.isMonthlySpecial,
        isCampaign: formData.isCampaign,
        campaignPrice: formData.campaignPrice,
        sortOrder: formData.order,  // Map order to sortOrder for API
        preparationTime: formData.preparationTime,
        allergens: formData.allergens
      };
      
      await apiRequest(`/api/menu-items/${selectedMenuItem.id}`, {
        method: "PATCH",
        body: JSON.stringify(menuItemData)
      });

      queryClient.invalidateQueries({ queryKey: ['/api/menu-items'] });
      queryClient.invalidateQueries({ queryKey: ['/api/menu'] });
      toast({
        title: "Menü öğesi güncellendi",
        description: `${formData.name} başarıyla güncellendi.`
      });
      setIsEditDialogOpen(false);
    } catch (error) {
      toast({
        title: "Hata",
        description: "Menü öğesi güncellenirken bir hata oluştu.",
        variant: "destructive"
      });
    }
  };

  // Delete menu item
  const handleDeleteMenuItem = async () => {
    if (!selectedMenuItem) return;
    
    try {
      await apiRequest(`/api/menu-items/${selectedMenuItem.id}`, {
        method: "DELETE"
      });

      queryClient.invalidateQueries({ queryKey: ['/api/menu-items'] });
      queryClient.invalidateQueries({ queryKey: ['/api/menu'] });
      toast({
        title: "Menü öğesi silindi",
        description: `${selectedMenuItem.name} başarıyla silindi.`
      });
      setIsDeleteDialogOpen(false);
    } catch (error) {
      toast({
        title: "Hata",
        description: "Menü öğesi silinirken bir hata oluştu.",
        variant: "destructive"
      });
    }
  };
  
  // Handle bulk price update
  const handleBulkPriceUpdate = async () => {
    try {
      // Call the bulk update API endpoint
      const response = await apiRequest("/api/menu-items/bulk-update", {
        method: "POST",
        body: JSON.stringify({
          categoryId: bulkUpdateCategory,
          operation: bulkUpdateOperation,
          percentage: bulkUpdatePercentage
        })
      });
      
      const result = await response.json();
      
      if (result.success) {
        // Refresh our data
        queryClient.invalidateQueries({ queryKey: ['/api/menu-items'] });
        queryClient.invalidateQueries({ queryKey: ['/api/menu'] });
        
        toast({
          title: "Fiyatlar güncellendi",
          description: `${result.updated} öğenin fiyatı ${bulkUpdateOperation === "increase" ? "artırıldı" : "azaltıldı"}.`
        });
        
        setIsBulkPriceUpdateOpen(false);
      } else {
        toast({
          title: "Uyarı",
          description: result.message || "İşlem sırasında bir sorun oluştu.",
          variant: "destructive"
        });
      }
    } catch (error) {
      console.error("Bulk update error:", error);
      toast({
        title: "Hata",
        description: "Fiyatlar güncellenirken bir hata oluştu.",
        variant: "destructive"
      });
    }
  };

  // Get category name by id
  const getCategoryName = (categoryId: number) => {
    const category = categories.find(c => c.id === categoryId);
    return category ? category.name : "Bilinmeyen Kategori";
  };

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <h2 className="text-2xl font-bold text-primary font-display">Menü Öğeleri</h2>
        <div className="flex gap-2">
          <Button 
            variant="outline" 
            onClick={() => setIsBulkPriceUpdateOpen(true)} 
            className="flex items-center gap-1"
          >
            <Percent className="h-4 w-4" /> Toplu Fiyat Güncelle
          </Button>
          <Button onClick={handleOpenAddDialog} className="flex items-center gap-1">
            <Plus className="h-4 w-4" /> Menü Öğesi Ekle
          </Button>
        </div>
      </div>
      
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        {/* Arama kutusu */}
        <div className="col-span-2">
          <Input
            placeholder="Menü öğesi ara..."
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
            className="w-full"
          />
        </div>
        
        {/* Kategori filtresi */}
        <div>
          <Select 
            value={selectedCategory.toString()} 
            onValueChange={(value) => setSelectedCategory(value === "all" ? "all" : parseInt(value))}
          >
            <SelectTrigger>
              <SelectValue placeholder="Kategori filtrele" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">Tüm Kategoriler</SelectItem>
              {categories.map(category => (
                <SelectItem key={category.id} value={category.id.toString()}>
                  {category.name}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Tüm Menü Öğeleri</CardTitle>
          <CardDescription>
            Menünüzdeki yemekleri, içecekleri ve diğer öğeleri yönetin.
          </CardDescription>
        </CardHeader>
        <CardContent>
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>Adı</TableHead>
                <TableHead>Kategori</TableHead>
                <TableHead>Fiyat</TableHead>
                <TableHead>Mevcut</TableHead>
                <TableHead>Özel</TableHead>
                <TableHead>Sıra</TableHead>
                <TableHead className="text-right">İşlemler</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {menuItems.length === 0 ? (
                <TableRow>
                  <TableCell colSpan={7} className="text-center py-4 text-gray-500">
                    Henüz menü öğesi bulunmuyor
                  </TableCell>
                </TableRow>
              ) : (
                menuItems
                  .filter(item => {
                    // Arama terimini kontrol et
                    const matchesSearch = searchTerm.trim() === '' || 
                      item.name.toLowerCase().includes(searchTerm.toLowerCase()) || 
                      (item.description && item.description.toLowerCase().includes(searchTerm.toLowerCase()));
                    
                    // Kategori filtresini kontrol et
                    const matchesCategory = selectedCategory === 'all' || item.categoryId === selectedCategory;
                    
                    return matchesSearch && matchesCategory;
                  })
                  .sort((a, b) => {
                    // Önce kategoriye göre sırala, sonra kategori içinde sortOrder'a göre
                    if (a.categoryId === b.categoryId) {
                      return (a.sortOrder || 0) - (b.sortOrder || 0);
                    }
                    return a.categoryId - b.categoryId;
                  })
                  .map((menuItem) => (
                    <TableRow key={menuItem.id}>
                      <TableCell className="font-medium">{menuItem.name}</TableCell>
                      <TableCell>{getCategoryName(menuItem.categoryId)}</TableCell>
                      <TableCell>{menuItem.price}</TableCell>
                      <TableCell>
                        {menuItem.isActive ? (
                          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                            Mevcut
                          </span>
                        ) : (
                          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">
                            Tükendi
                          </span>
                        )}
                      </TableCell>
                      <TableCell>
                        {menuItem.isChefSpecial && (
                          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-primary-100 text-primary-800 mr-1">
                            Şef
                          </span>
                        )}
                        {menuItem.isMonthlySpecial && (
                          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-purple-100 text-purple-800 mr-1">
                            Ay
                          </span>
                        )}
                        {menuItem.isCampaign && (
                          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-orange-100 text-orange-800">
                            Kampanya
                          </span>
                        )}
                      </TableCell>
                      <TableCell>{menuItem.sortOrder || 0}</TableCell>
                      <TableCell className="text-right">
                        <div className="flex justify-end gap-2">
                          <Button 
                            variant="ghost" 
                            size="sm"
                            onClick={() => handleOpenEditDialog(menuItem)}
                          >
                            <Edit className="h-4 w-4" />
                          </Button>
                          <Button 
                            variant="ghost" 
                            size="sm" 
                            className="text-red-500 hover:text-red-700 hover:bg-red-50"
                            onClick={() => handleOpenDeleteDialog(menuItem)}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
              )}
            </TableBody>
          </Table>
        </CardContent>
      </Card>

      {/* Add Menu Item Dialog */}
      <Dialog open={isAddDialogOpen} onOpenChange={setIsAddDialogOpen}>
        <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto" aria-describedby="add-menu-item-description">
          <DialogHeader>
            <DialogTitle>Yeni Menü Öğesi Ekle</DialogTitle>
            <DialogClose className="absolute right-4 top-4">
              <X className="h-4 w-4" />
            </DialogClose>
            <p id="add-menu-item-description" className="text-sm text-muted-foreground">
              Yeni bir menü öğesi eklemek için aşağıdaki alanları doldurun.
            </p>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <label htmlFor="name">Adı</label>
                <Input
                  id="name"
                  name="name"
                  value={formData.name}
                  onChange={handleInputChange}
                />
              </div>
              <div className="space-y-2">
                <label htmlFor="price">Fiyat</label>
                <Input
                  id="price"
                  name="price"
                  value={formData.price}
                  onChange={handleInputChange}
                  placeholder="29.99 TL"
                />
              </div>
              <div className="space-y-2">
                <label htmlFor="categoryId">Kategori</label>
                <Select 
                  value={formData.categoryId.toString()} 
                  onValueChange={(value) => handleSelectChange("categoryId", value)}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Kategori seçin" />
                  </SelectTrigger>
                  <SelectContent>
                    {categories.map(category => (
                      <SelectItem key={category.id} value={category.id.toString()}>
                        {category.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <label htmlFor="order">Sıralama</label>
                <Input
                  id="order"
                  name="order"
                  type="number"
                  value={formData.order}
                  onChange={handleInputChange}
                />
              </div>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <label htmlFor="preparationTime">Hazırlık Süresi</label>
                <Input
                  id="preparationTime"
                  name="preparationTime"
                  value={formData.preparationTime || ""}
                  onChange={handleInputChange}
                  placeholder="15-20 dakika"
                />
              </div>
              <div className="space-y-2">
                <label htmlFor="allergens">Alerjenler</label>
                <Input
                  id="allergens"
                  name="allergens"
                  value={formData.allergens || ""}
                  onChange={handleInputChange}
                  placeholder="Süt, yumurta, gluten"
                />
              </div>
            </div>
            
            <div className="space-y-2">
              <label htmlFor="description">Açıklama</label>
              <Textarea
                id="description"
                name="description"
                value={formData.description}
                onChange={handleInputChange}
                rows={3}
              />
            </div>
            
            <div className="space-y-4">
              <div>
                <label htmlFor="imageUrl" className="block mb-2">Ana Resim URL'i</label>
                <div className="flex gap-2">
                  <Input
                    id="imageUrl"
                    name="imageUrl"
                    value={formData.imageUrl || ""}
                    onChange={handleInputChange}
                    placeholder="https://example.com/image.jpg"
                    className="flex-1"
                  />
                  <Button 
                    variant="outline" 
                    size="icon" 
                    className="shrink-0"
                    type="button"
                    onClick={handleImagePreview}
                  >
                    <Image className="h-4 w-4" />
                  </Button>
                </div>
              </div>
              
              <div>
                <label htmlFor="file-upload" className="block mb-2">veya Bilgisayardan Yükle</label>
                <Input
                  id="file-upload"
                  type="file"
                  accept="image/*"
                  ref={fileInputRef}
                  onChange={handleFileUpload}
                  className="w-full"
                />
              </div>
              
              {formData.imageUrl && (
                <div className="mt-2 p-3 bg-gray-50 rounded-md border border-gray-100">
                  <div className="relative bg-white border rounded-md overflow-hidden shadow-sm hover:shadow-md transition h-32">
                    <img 
                      src={formData.imageUrl} 
                      alt="Ana görsel önizleme" 
                      className="w-full h-full object-cover"
                      onError={(e) => {
                        (e.target as HTMLImageElement).src = 'https://via.placeholder.com/200x120?text=Görsel+Yüklenemedi';
                      }}
                    />
                    {formData.imageUrl && (
                      <div className="absolute bottom-0 left-0 right-0 bg-black/50 text-white text-xs px-2 py-1">
                        Ana Görsel
                      </div>
                    )}
                  </div>
                </div>
              )}
              
              <p className="text-xs text-gray-500">
                Ürün için ana görsel olarak kullanılacak resim URL'i girin.
              </p>
            </div>
            
            <div className="space-y-2">
              <label htmlFor="additionalImages">Ek Resim URL'leri</label>
              <div className="space-y-2">
                <Textarea
                  id="additionalImages"
                  name="additionalImages"
                  value={formData.additionalImages || ""}
                  onChange={handleInputChange}
                  placeholder="https://example.com/image2.jpg,https://example.com/image3.jpg"
                  rows={3}
                />
                
                {formData.additionalImages && (
                  <div className="mt-2 p-3 bg-gray-50 rounded-md border border-gray-100">
                    <h4 className="text-sm font-medium text-gray-700 mb-2 flex items-center">
                      <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4 mr-1 text-accent" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
                      </svg>
                      Galeri Görselleri Önizleme
                    </h4>
                    <div className="grid grid-cols-3 gap-2">
                      {formData.additionalImages.split(',').filter(url => url.trim()).map((url, index) => (
                        <div key={index} className="relative bg-white border rounded-md overflow-hidden shadow-sm group hover:shadow-md transition">
                          <img 
                            src={url} 
                            alt={`Ek görsel ${index + 1}`} 
                            className="w-full h-16 object-cover"
                            onError={(e) => {
                              (e.target as HTMLImageElement).src = 'https://via.placeholder.com/200x120?text=Görsel+Yüklenemedi';
                            }}
                          />
                          <div className="absolute inset-0 bg-black/30 opacity-0 group-hover:opacity-100 transition flex items-center justify-center">
                            <button 
                              type="button"
                              className="p-1 bg-red-500 text-white rounded-full"
                              onClick={() => {
                                const urls = formData.additionalImages.split(',').filter(u => u.trim());
                                urls.splice(index, 1);
                                setFormData({
                                  ...formData,
                                  additionalImages: urls.join(',')
                                });
                              }}
                            >
                              <X className="h-3 w-3" />
                            </button>
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>
                )}
              </div>
              <p className="text-xs text-gray-500">
                Birden fazla ek görsel için URL'leri virgülle ayırarak girin. Bu görseller ürün detay sayfasında kaydırmalı galeri olarak gösterilecektir.
              </p>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-4">
                <h4 className="text-sm font-semibold border-b pb-2">Öğe Durumu</h4>
                <div className="flex items-center space-x-2">
                  <Switch
                    id="isAvailable"
                    checked={formData.isAvailable}
                    onCheckedChange={checked => handleSwitchChange("isAvailable", checked)}
                  />
                  <Label htmlFor="isAvailable">Sipariş verilebilir</Label>
                </div>
                
                <div className="flex items-center space-x-2">
                  <Switch
                    id="isVegetarian"
                    checked={formData.isVegetarian}
                    onCheckedChange={checked => handleSwitchChange("isVegetarian", checked)}
                  />
                  <Label htmlFor="isVegetarian">Vejetaryen</Label>
                </div>
                
                <div className="flex items-center space-x-2">
                  <Switch
                    id="isSpicy"
                    checked={formData.isSpicy}
                    onCheckedChange={checked => handleSwitchChange("isSpicy", checked)}
                  />
                  <Label htmlFor="isSpicy">Acı</Label>
                </div>
              </div>
              
              <div className="space-y-4">
                <h4 className="text-sm font-semibold border-b pb-2">Özel Menüler</h4>
                <div className="flex items-center space-x-2">
                  <Switch
                    id="isChefSpecial"
                    checked={formData.isChefSpecial}
                    onCheckedChange={checked => handleSwitchChange("isChefSpecial", checked)}
                  />
                  <Label htmlFor="isChefSpecial">Şefin Spesiyali</Label>
                </div>
                
                <div className="flex items-center space-x-2">
                  <Switch
                    id="isMonthlySpecial"
                    checked={formData.isMonthlySpecial}
                    onCheckedChange={checked => handleSwitchChange("isMonthlySpecial", checked)}
                  />
                  <Label htmlFor="isMonthlySpecial">Ayın Spesiyali</Label>
                </div>
                
                <div className="flex items-center space-x-2">
                  <Switch
                    id="isCampaign"
                    checked={formData.isCampaign}
                    onCheckedChange={checked => handleSwitchChange("isCampaign", checked)}
                  />
                  <Label htmlFor="isCampaign">Kampanya</Label>
                </div>
                
                {formData.isCampaign && (
                  <div className="space-y-2 mt-2">
                    <Label htmlFor="campaignPrice">Kampanya Fiyatı</Label>
                    <Input
                      id="campaignPrice"
                      name="campaignPrice"
                      value={formData.campaignPrice || ""}
                      onChange={handleInputChange}
                      placeholder="99.90 ₺"
                    />
                  </div>
                )}
              </div>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setIsAddDialogOpen(false)}>
              İptal
            </Button>
            <Button onClick={handleAddMenuItem}>
              Menü Öğesi Ekle
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Edit Menu Item Dialog */}
      <Dialog open={isEditDialogOpen} onOpenChange={setIsEditDialogOpen}>
        <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto" aria-describedby="edit-menu-item-description">
          <DialogHeader>
            <DialogTitle>Menü Öğesini Düzenle</DialogTitle>
            <DialogClose className="absolute right-4 top-4">
              <X className="h-4 w-4" />
            </DialogClose>
            <p id="edit-menu-item-description" className="text-sm text-muted-foreground">
              Menü öğesinin bilgilerini düzenleyebilirsiniz.
            </p>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <label htmlFor="edit-name">Adı</label>
                <Input
                  id="edit-name"
                  name="name"
                  value={formData.name}
                  onChange={handleInputChange}
                />
              </div>
              <div className="space-y-2">
                <label htmlFor="edit-price">Fiyat</label>
                <Input
                  id="edit-price"
                  name="price"
                  value={formData.price}
                  onChange={handleInputChange}
                />
              </div>
              <div className="space-y-2">
                <label htmlFor="edit-categoryId">Kategori</label>
                <Select 
                  value={formData.categoryId.toString()} 
                  onValueChange={(value) => handleSelectChange("categoryId", value)}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Kategori seçin" />
                  </SelectTrigger>
                  <SelectContent>
                    {categories.map(category => (
                      <SelectItem key={category.id} value={category.id.toString()}>
                        {category.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <label htmlFor="edit-order">Sıralama</label>
                <Input
                  id="edit-order"
                  name="order"
                  type="number"
                  value={formData.order}
                  onChange={handleInputChange}
                />
              </div>
            </div>
            
            <div className="space-y-2">
              <label htmlFor="edit-description">Açıklama</label>
              <Textarea
                id="edit-description"
                name="description"
                value={formData.description}
                onChange={handleInputChange}
                rows={3}
              />
            </div>
            
            <div className="space-y-4">
              <div>
                <label htmlFor="edit-imageUrl" className="block mb-2">Ana Resim URL'i</label>
                <div className="flex gap-2">
                  <Input
                    id="edit-imageUrl"
                    name="imageUrl"
                    value={formData.imageUrl || ""}
                    onChange={handleInputChange}
                    placeholder="https://example.com/image.jpg"
                    className="flex-1"
                  />
                  <Button 
                    variant="outline" 
                    size="icon" 
                    className="shrink-0"
                    type="button"
                    onClick={handleImagePreview}
                  >
                    <Image className="h-4 w-4" />
                  </Button>
                </div>
              </div>
              
              <div>
                <label htmlFor="edit-file-upload" className="block mb-2">veya Bilgisayardan Yükle</label>
                <Input
                  id="edit-file-upload"
                  type="file"
                  accept="image/*"
                  ref={fileInputRef}
                  onChange={handleFileUpload}
                  className="w-full"
                />
              </div>
              
              {formData.imageUrl && (
                <div className="mt-2 p-3 bg-gray-50 rounded-md border border-gray-100">
                  <div className="relative bg-white border rounded-md overflow-hidden shadow-sm hover:shadow-md transition h-32">
                    <img 
                      src={formData.imageUrl} 
                      alt="Ana görsel önizleme" 
                      className="w-full h-full object-cover"
                      onError={(e) => {
                        (e.target as HTMLImageElement).src = 'https://via.placeholder.com/200x120?text=Görsel+Yüklenemedi';
                      }}
                    />
                    {formData.imageUrl && (
                      <div className="absolute bottom-0 left-0 right-0 bg-black/50 text-white text-xs px-2 py-1">
                        Ana Görsel
                      </div>
                    )}
                  </div>
                </div>
              )}
            </div>
            
            <div className="space-y-2">
              <label htmlFor="edit-additionalImages">Ek Resim URL'leri</label>
              <div className="space-y-2">
                <Textarea
                  id="edit-additionalImages"
                  name="additionalImages"
                  value={formData.additionalImages || ""}
                  onChange={handleInputChange}
                  placeholder="https://example.com/image2.jpg,https://example.com/image3.jpg"
                  rows={3}
                />
                
                {formData.additionalImages && (
                  <div className="mt-2 p-3 bg-gray-50 rounded-md border border-gray-100">
                    <h4 className="text-sm font-medium text-gray-700 mb-2 flex items-center">
                      <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4 mr-1 text-accent" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
                      </svg>
                      Galeri Görselleri Önizleme
                    </h4>
                    <div className="grid grid-cols-3 gap-2">
                      {formData.additionalImages.split(',').filter(url => url.trim()).map((url, index) => (
                        <div key={index} className="relative bg-white border rounded-md overflow-hidden shadow-sm group hover:shadow-md transition">
                          <img 
                            src={url} 
                            alt={`Ek görsel ${index + 1}`} 
                            className="w-full h-16 object-cover"
                            onError={(e) => {
                              (e.target as HTMLImageElement).src = 'https://via.placeholder.com/200x120?text=Görsel+Yüklenemedi';
                            }}
                          />
                          <div className="absolute inset-0 bg-black/30 opacity-0 group-hover:opacity-100 transition flex items-center justify-center">
                            <button 
                              type="button"
                              className="p-1 bg-red-500 text-white rounded-full"
                              onClick={() => {
                                const urls = formData.additionalImages.split(',').filter(u => u.trim());
                                urls.splice(index, 1);
                                setFormData({
                                  ...formData,
                                  additionalImages: urls.join(',')
                                });
                              }}
                              aria-label={`Ek görsel ${index + 1} sil`}
                            >
                              <X className="h-3 w-3" />
                            </button>
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>
                )}
              </div>
              <p className="text-xs text-gray-500">
                Birden fazla ek görsel için URL'leri virgülle ayırarak girin. Bu görseller ürün detay sayfasında kaydırmalı galeri olarak gösterilecektir.
              </p>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-4">
                <h4 className="text-sm font-semibold border-b pb-2">Öğe Durumu</h4>
                <div className="flex items-center space-x-2">
                  <Switch
                    id="edit-isAvailable"
                    checked={formData.isAvailable}
                    onCheckedChange={checked => handleSwitchChange("isAvailable", checked)}
                  />
                  <Label htmlFor="edit-isAvailable">Sipariş verilebilir</Label>
                </div>
                
                <div className="flex items-center space-x-2">
                  <Switch
                    id="edit-isVegetarian"
                    checked={formData.isVegetarian}
                    onCheckedChange={checked => handleSwitchChange("isVegetarian", checked)}
                  />
                  <Label htmlFor="edit-isVegetarian">Vejetaryen</Label>
                </div>
                
                <div className="flex items-center space-x-2">
                  <Switch
                    id="edit-isSpicy"
                    checked={formData.isSpicy}
                    onCheckedChange={checked => handleSwitchChange("isSpicy", checked)}
                  />
                  <Label htmlFor="edit-isSpicy">Acı</Label>
                </div>
              </div>
              
              <div className="space-y-4">
                <h4 className="text-sm font-semibold border-b pb-2">Özel Menüler</h4>
                <div className="flex items-center space-x-2">
                  <Switch
                    id="edit-isChefSpecial"
                    checked={formData.isChefSpecial}
                    onCheckedChange={checked => handleSwitchChange("isChefSpecial", checked)}
                  />
                  <Label htmlFor="edit-isChefSpecial">Şefin Spesiyali</Label>
                </div>
                
                <div className="flex items-center space-x-2">
                  <Switch
                    id="edit-isMonthlySpecial"
                    checked={formData.isMonthlySpecial}
                    onCheckedChange={checked => handleSwitchChange("isMonthlySpecial", checked)}
                  />
                  <Label htmlFor="edit-isMonthlySpecial">Ayın Spesiyali</Label>
                </div>
                
                <div className="flex items-center space-x-2">
                  <Switch
                    id="edit-isCampaign"
                    checked={formData.isCampaign}
                    onCheckedChange={checked => handleSwitchChange("isCampaign", checked)}
                  />
                  <Label htmlFor="edit-isCampaign">Kampanya</Label>
                </div>
                
                {formData.isCampaign && (
                  <div className="space-y-2 mt-2">
                    <Label htmlFor="edit-campaignPrice">Kampanya Fiyatı</Label>
                    <Input
                      id="edit-campaignPrice"
                      name="campaignPrice"
                      value={formData.campaignPrice || ""}
                      onChange={handleInputChange}
                      placeholder="99.90 ₺"
                    />
                  </div>
                )}
              </div>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setIsEditDialogOpen(false)}>
              İptal
            </Button>
            <Button onClick={handleUpdateMenuItem}>
              Güncelle
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Delete Menu Item Dialog */}
      <Dialog open={isDeleteDialogOpen} onOpenChange={setIsDeleteDialogOpen}>
        <DialogContent className="max-w-md" aria-describedby="delete-menu-item-description">
          <DialogHeader>
            <DialogTitle>Menü Öğesini Sil</DialogTitle>
            <DialogClose className="absolute right-4 top-4">
              <X className="h-4 w-4" aria-hidden="true" />
              <span className="sr-only">Kapat</span>
            </DialogClose>
            <p id="delete-menu-item-description" className="text-sm text-muted-foreground">
              Bu işlem geri alınamaz.
            </p>
          </DialogHeader>
          <div className="py-4">
            {selectedMenuItem?.image && (
              <div className="mb-4 flex justify-center">
                <div className="relative w-32 h-32 border rounded-md overflow-hidden shadow-sm">
                  <img 
                    src={selectedMenuItem.image} 
                    alt={`${selectedMenuItem.name} görsel`} 
                    className="w-full h-full object-cover"
                    onError={(e) => {
                      (e.target as HTMLImageElement).src = 'https://via.placeholder.com/200x120?text=Görsel+Yüklenemedi';
                    }}
                  />
                </div>
              </div>
            )}
            <div className="bg-red-50 border border-red-100 rounded-md p-4 mb-4">
              <div className="flex items-center">
                <div className="mr-3 text-red-500">
                  <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-5 w-5">
                    <circle cx="12" cy="12" r="10"></circle>
                    <line x1="12" y1="8" x2="12" y2="12"></line>
                    <line x1="12" y1="16" x2="12.01" y2="16"></line>
                  </svg>
                </div>
                <div className="text-sm text-red-600">
                  <p className="font-medium">
                    <strong>{selectedMenuItem?.name}</strong> menü öğesini silmek istediğinizden emin misiniz?
                  </p>
                  <p className="mt-1">
                    Bu işlem geri alınamaz ve tüm veriler kalıcı olarak silinecektir.
                  </p>
                </div>
              </div>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setIsDeleteDialogOpen(false)}>
              İptal
            </Button>
            <Button 
              variant="destructive" 
              onClick={handleDeleteMenuItem}
              aria-label={`${selectedMenuItem?.name} menü öğesini sil`}
            >
              Sil
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
      
      {/* Bulk Price Update Dialog */}
      <AlertDialog open={isBulkPriceUpdateOpen} onOpenChange={setIsBulkPriceUpdateOpen}>
        <AlertDialogContent className="max-w-lg">
          <AlertDialogHeader>
            <AlertDialogTitle>Toplu Fiyat Güncelleme</AlertDialogTitle>
            <AlertDialogDescription>
              Seçilen kategorideki tüm ürünlerin fiyatlarını belirtilen oranda artırabilir veya azaltabilirsiniz.
            </AlertDialogDescription>
          </AlertDialogHeader>
          
          <div className="grid gap-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="bulkUpdateCategory">Kategori Seçin</Label>
              <Select
                value={bulkUpdateCategory.toString()}
                onValueChange={(value) => setBulkUpdateCategory(value === "all" ? "all" : parseInt(value))}
              >
                <SelectTrigger id="bulkUpdateCategory">
                  <SelectValue placeholder="Kategori seçin" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Tüm Kategoriler</SelectItem>
                  {categories.map(category => (
                    <SelectItem key={category.id} value={category.id.toString()}>
                      {category.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="bulkUpdateOperation">İşlem</Label>
              <Select
                value={bulkUpdateOperation}
                onValueChange={(value) => setBulkUpdateOperation(value as "increase" | "decrease")}
              >
                <SelectTrigger id="bulkUpdateOperation">
                  <SelectValue placeholder="İşlem seçin" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="increase">Fiyat Artışı</SelectItem>
                  <SelectItem value="decrease">Fiyat İndirimi</SelectItem>
                </SelectContent>
              </Select>
            </div>
            
            <div className="space-y-2">
              <div className="flex justify-between items-center">
                <Label htmlFor="bulkUpdatePercentage">Oran (%)</Label>
                <span className="text-sm font-medium">{bulkUpdatePercentage}%</span>
              </div>
              <Slider
                id="bulkUpdatePercentage"
                min={1}
                max={50}
                step={1}
                value={[bulkUpdatePercentage]}
                onValueChange={(values) => setBulkUpdatePercentage(values[0])}
                className="py-4"
              />
              <div className="flex justify-between text-xs text-muted-foreground px-1">
                <span>1%</span>
                <span>25%</span>
                <span>50%</span>
              </div>
            </div>
            
            <div className="bg-muted p-3 rounded-md mt-2">
              <h4 className="font-medium mb-2 text-sm">Önizleme:</h4>
              <p className="text-sm">
                {bulkUpdateCategory === "all" ? "Tüm menü öğelerinin" : `"${getCategoryName(+bulkUpdateCategory)}" kategorisindeki öğelerin`} fiyatları
                <strong> %{bulkUpdatePercentage} oranında {bulkUpdateOperation === "increase" ? "artırılacak" : "azaltılacak"}</strong>.
              </p>
              <p className="text-sm mt-2">
                Örnek: <span className="font-mono">100 TL</span> → 
                <span className="font-mono font-medium"> {bulkUpdateOperation === "increase" 
                  ? (100 * (1 + bulkUpdatePercentage / 100)).toFixed(2) 
                  : (100 * (1 - bulkUpdatePercentage / 100)).toFixed(2)} TL</span>
              </p>
            </div>
          </div>
          
          <AlertDialogFooter>
            <AlertDialogCancel asChild>
              <Button variant="outline">İptal</Button>
            </AlertDialogCancel>
            <AlertDialogAction asChild>
              <Button onClick={handleBulkPriceUpdate}>Fiyatları Güncelle</Button>
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
};