import React, { useState, useRef } from "react";
import {
  Card,
  CardContent,
  CardDescription,
  CardFooter,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Loader2, Download, Save, Upload, AlertTriangle, Image as ImageIcon } from "lucide-react";
import { useMutation } from "@tanstack/react-query";
import { queryClient } from "@/lib/queryClient";
import { toast } from "@/hooks/use-toast";
import { Progress } from "@/components/ui/progress";
import { useMenuItems } from "@/hooks/use-menu-items";
import { Image } from "@shared/schema";

export function ImageImporter() {
  const [imageUrl, setImageUrl] = useState("");
  const [previewUrls, setPreviewUrls] = useState<string[]>([]);
  const [selectedMenuItemId, setSelectedMenuItemId] = useState<number | null>(null);
  const [isImporting, setIsImporting] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);
  const fileInputRef = useRef<HTMLInputElement>(null);
  const { data: menuItems } = useMenuItems();

  // URL'den görseli önizle
  const previewImageFromUrl = () => {
    if (!imageUrl) {
      toast({
        title: "URL gerekli",
        description: "Lütfen bir görsel URL'si girin",
        variant: "destructive",
      });
      return;
    }

    setPreviewUrls([imageUrl]);
    toast({
      title: "Görsel yüklendi",
      description: "Görsel başarıyla önizlendi. Şimdi kaydedebilirsiniz.",
    });
  };

  // Dosyadan görseli önizle
  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const files = e.target.files;
    if (!files || files.length === 0) return;

    const newPreviewUrls: string[] = [];
    
    for (let i = 0; i < files.length; i++) {
      const file = files[i];
      if (!file.type.startsWith("image/")) continue;
      
      const reader = new FileReader();
      reader.onload = (e) => {
        if (e.target?.result) {
          newPreviewUrls.push(e.target.result as string);
          
          // Tüm dosyalar yüklenince state'i güncelle
          if (newPreviewUrls.length === Math.min(files.length, 10)) {
            setPreviewUrls(newPreviewUrls);
          }
        }
      };
      
      reader.readAsDataURL(file);
      
      // Maksimum 10 görsel
      if (i >= 9) break;
    }
  };

  // URL'den görsel yükle
  const uploadImageFromUrlMutation = useMutation({
    mutationFn: async () => {
      // Giriş kontrolü
      if (previewUrls.length === 0) {
        throw new Error("Yüklenecek görsel bulunamadı. Lütfen URL girin veya dosya yükleyin.");
      }
      
      if (!selectedMenuItemId) {
        throw new Error("Lütfen bir menü öğesi seçin");
      }
      
      setIsImporting(true);
      setUploadProgress(10);
      
      // Seçilen URL (şu an sadece ilk URL'yi kullanıyoruz)
      const selectedUrl = previewUrls[0];
      
      try {
        // 1. Önce görsel oluştur
        console.log("Görsel oluşturuluyor:", selectedUrl);
        const createImageRes = await fetch("/api/images", {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
          },
          body: JSON.stringify({
            url: selectedUrl,
            title: "Menü Görseli",
            altText: "Menü öğesi görseli"
          }),
          credentials: "include"
        });
        
        if (!createImageRes.ok) {
          const errorText = await createImageRes.text();
          console.error("Görsel oluşturma hatası:", errorText);
          throw new Error(`Görsel oluşturulamadı: ${errorText}`);
        }
        
        setUploadProgress(50);
        
        const image: Image = await createImageRes.json();
        console.log("Oluşturulan görsel:", image);
        
        // 2. Görseli menü öğesine bağla
        console.log("Görsel menü öğesine bağlanıyor:", selectedMenuItemId, image.id);
        const addToMenuRes = await fetch(`/api/menu-items/${selectedMenuItemId}/images/${image.id}`, {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
          },
          body: JSON.stringify({ order: 0 }),
          credentials: "include"
        });
        
        if (!addToMenuRes.ok) {
          const errorText = await addToMenuRes.text();
          console.error("Menü öğesine ekleme hatası:", errorText);
          throw new Error(`Görsel menü öğesine eklenemedi: ${errorText}`);
        }
        
        console.log("Görsel başarıyla eklendi");
        setUploadProgress(100);
        
        return image;
      } catch (error) {
        console.error("Görsel yükleme hatası:", error);
        throw error;
      } finally {
        setIsImporting(false);
      }
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/images'] });
      queryClient.invalidateQueries({ queryKey: ['/api/menu-items'] });
      
      toast({
        title: "Görsel eklendi",
        description: "Görsel başarıyla menü öğesine eklendi.",
      });
      
      // Form temizle
      setImageUrl("");
      setPreviewUrls([]);
      setSelectedMenuItemId(null);
      setUploadProgress(0);
      
      if (fileInputRef.current) {
        fileInputRef.current.value = "";
      }
    },
    onError: (error: Error) => {
      setIsImporting(false);
      setUploadProgress(0);
      
      toast({
        title: "Görsel eklenemedi",
        description: error.message,
        variant: "destructive",
      });
    }
  });

  return (
    <Card className="w-full">
      <CardHeader>
        <CardTitle>Görselleri İçe Aktar</CardTitle>
        <CardDescription>
          Menü öğelerine görsel eklemek için URL'den veya bilgisayarınızdan görsel yükleyebilirsiniz.
        </CardDescription>
      </CardHeader>
      
      <CardContent className="space-y-4">
        {/* Menü öğesi seçimi */}
        <div className="space-y-2">
          <Label htmlFor="menu-item">Menü Öğesi</Label>
          <select
            id="menu-item"
            className="w-full p-2 border rounded-md"
            value={selectedMenuItemId || ""}
            onChange={(e) => setSelectedMenuItemId(Number(e.target.value) || null)}
          >
            <option value="">Menü öğesi seçin</option>
            {menuItems?.map((item) => (
              <option key={item.id} value={item.id}>
                {item.name}
              </option>
            ))}
          </select>
        </div>
        
        {/* URL ile yükleme */}
        <div className="space-y-2 border-t pt-4">
          <Label htmlFor="image-url">Görsel URL'si</Label>
          <div className="flex space-x-2">
            <Input
              id="image-url"
              type="url"
              placeholder="https://example.com/image.jpg"
              value={imageUrl}
              onChange={(e) => setImageUrl(e.target.value)}
            />
            <Button 
              variant="outline" 
              onClick={previewImageFromUrl}
              disabled={!imageUrl}
            >
              <Download className="mr-2 h-4 w-4" />
              Önizle
            </Button>
          </div>
          <p className="text-xs text-muted-foreground">
            Doğrudan bir görsel URL'si girin, örneğin: https://jouskamenu.com/img/menu/kahvalti1.jpg
          </p>
        </div>
        
        {/* Dosyadan yükleme */}
        <div className="space-y-2 border-t pt-4">
          <Label htmlFor="file-upload">Bilgisayardan Yükle</Label>
          <Input
            id="file-upload"
            type="file"
            accept="image/*"
            ref={fileInputRef}
            onChange={handleFileChange}
            multiple
          />
        </div>
        
        {/* Önizleme */}
        {previewUrls.length > 0 && (
          <div className="space-y-2 border-t pt-4">
            <Label>Önizleme</Label>
            <div className="grid grid-cols-2 gap-2">
              {previewUrls.map((url, index) => (
                <div key={index} className="relative aspect-video rounded-md overflow-hidden border">
                  <img
                    src={url}
                    alt={`Preview ${index + 1}`}
                    className="w-full h-full object-cover"
                    onError={(e) => {
                      e.currentTarget.src = "https://via.placeholder.com/300x200?text=Görsel+Yüklenemedi";
                    }}
                  />
                </div>
              ))}
            </div>
          </div>
        )}
        
        {/* Hata durumu */}
        {uploadImageFromUrlMutation.isError && (
          <div className="text-sm text-red-500 mt-2">
            <p>Hata: {uploadImageFromUrlMutation.error instanceof Error ? uploadImageFromUrlMutation.error.message : "Bilinmeyen hata"}</p>
            <p className="text-xs mt-1">
              Görsel URL'sinin doğru olduğundan ve doğrudan bir görsele işaret ettiğinden emin olun.
              <br />
              Örnek bir doğru URL: https://jouskamenu.com/img/menu/kahvalti1.jpg
            </p>
          </div>
        )}
        
        {/* Yükleme durumu */}
        {isImporting && (
          <div className="space-y-2 border-t pt-4">
            <Label>Yükleniyor...</Label>
            <Progress value={uploadProgress} className="h-2" />
          </div>
        )}
      </CardContent>
      
      <CardFooter className="flex justify-between">
        <Button 
          variant="outline"
          onClick={() => {
            setImageUrl("");
            setPreviewUrls([]);
            setSelectedMenuItemId(null);
            setUploadProgress(0);
            if (fileInputRef.current) {
              fileInputRef.current.value = "";
            }
          }}
          disabled={isImporting}
        >
          Temizle
        </Button>
        
        <Button
          onClick={() => uploadImageFromUrlMutation.mutate()}
          disabled={!selectedMenuItemId || previewUrls.length === 0 || isImporting}
        >
          {isImporting ? (
            <>
              <Loader2 className="mr-2 h-4 w-4 animate-spin" />
              Yükleniyor...
            </>
          ) : (
            <>
              <ImageIcon className="mr-2 h-4 w-4" />
              Görseli Ekle
            </>
          )}
        </Button>
      </CardFooter>
    </Card>
  );
}