import React, { useState, useEffect } from "react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { useQueryClient } from "@tanstack/react-query";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { 
  Loader2, 
  MessageCircle, 
  Plus, 
  X, 
  Save,
  RotateCcw,
  Settings,
  Zap,
  Bot,
  MessageSquare,
  ShoppingCart,
  Calendar,
  Share2,
  BarChart3,
  Edit
} from "lucide-react";

interface QnAPair {
  id: string;
  question: string;
  answer: string;
  keywords: string[];
  isActive: boolean;
}

interface ChatbotConfig {
  isEnabled: boolean;
  welcomeMessage: string;
  autoOpen: boolean;
  openDelay: number;
  quickReplies: string[];
  categories: string[];
  searchKeywords: string[];
  botName: string;
  botAvatar: string;
  themeColor: string;
  qnaPairs: QnAPair[];
  customerQuestions: CustomerQuestion[];
  // Yeni özellikler
  features: {
    orderSystem: {
      enabled: boolean;
      minOrderAmount: number;
      deliveryFee: number;
      estimatedDeliveryTime: string;
      paymentMethods: string[];
    };
    reservationSystem: {
      enabled: boolean;
      maxAdvanceDays: number;
      timeSlots: string[];
      maxPartySize: number;
      requirePhone: boolean;
    };
    socialFeatures: {
      whatsappEnabled: boolean;
      whatsappNumber: string;
      instagramEnabled: boolean;
      instagramHandle: string;
      reviewsEnabled: boolean;
    };
    analytics: {
      enabled: boolean;
      trackPopularItems: boolean;
      trackCustomerBehavior: boolean;
      generateReports: boolean;
    };
  };
}

interface CustomerQuestion {
  id: string;
  question: string;
  timestamp: string;
  hasAnswer: boolean;
  answerGiven?: string;
  frequency: number;
}

export function ChatbotSettings() {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [isLoading, setIsLoading] = useState(true);
  const [isSaving, setIsSaving] = useState(false);
  const [newQuickReply, setNewQuickReply] = useState("");
  const [newKeyword, setNewKeyword] = useState("");
  const [newQuestion, setNewQuestion] = useState("");
  const [newAnswer, setNewAnswer] = useState("");
  const [newQnaKeywords, setNewQnaKeywords] = useState("");
  const [editingQna, setEditingQna] = useState<QnAPair | null>(null);

  const [config, setConfig] = useState<ChatbotConfig>({
    isEnabled: true,
    welcomeMessage: "Merhaba, Jouska Cafe & Restaurant'ya hoş geldiniz! 🌟 Size nasıl yardımcı olabilirim? Menüye göz atmak veya arama yaparak istediğiniz lezzetleri daha hızlı bulmanız için buradayım. Ne arıyorsunuz?",
    autoOpen: true,
    openDelay: 1500,
    quickReplies: [
      "Menüyü görüntüle",
      "Kahvaltılar", 
      "Ana Yemekler",
      "Popüler Ürünler",
      "Vejetaryen",
      "Hızlı Servis"
    ],
    categories: ["kahvaltilar", "ana-yemekler", "icecekler", "tatlilar"],
    searchKeywords: ["popüler", "vejetaryen", "hızlı", "sıcak", "soğuk", "taze"],
    botName: "Jouska Asistan",
    botAvatar: "🤖",
    themeColor: "#f97316",
    qnaPairs: [
      {
        id: "1",
        question: "Açılış saatleriniz nedir?",
        answer: "Haftanın her günü 08:00-23:00 saatleri arasında hizmet veriyoruz.",
        keywords: ["açılış", "saat", "zaman", "kaçta", "kapanış"],
        isActive: true
      },
      {
        id: "2", 
        question: "Teslimat yapıyor musunuz?",
        answer: "Evet, online sipariş ile teslimat hizmeti sunuyoruz. Minimum sipariş tutarı 50₺'dir.",
        keywords: ["teslimat", "sipariş", "online", "getir"],
        isActive: true
      }
    ],
    customerQuestions: [],
    features: {
      orderSystem: {
        enabled: false,
        minOrderAmount: 50,
        deliveryFee: 15,
        estimatedDeliveryTime: "30-45 dakika",
        paymentMethods: ["Nakit", "Kredi Kartı", "Online Ödeme"]
      },
      reservationSystem: {
        enabled: false,
        maxAdvanceDays: 30,
        timeSlots: ["12:00", "12:30", "13:00", "13:30", "19:00", "19:30", "20:00", "20:30"],
        maxPartySize: 8,
        requirePhone: true
      },
      socialFeatures: {
        whatsappEnabled: false,
        whatsappNumber: "",
        instagramEnabled: false,
        instagramHandle: "",
        reviewsEnabled: false
      },
      analytics: {
        enabled: true,
        trackPopularItems: true,
        trackCustomerBehavior: true,
        generateReports: false
      }
    }
  });

  // Load chatbot configuration
  useEffect(() => {
    const loadConfig = async () => {
      try {
        const savedConfig = localStorage.getItem('chatbotConfig');
        if (savedConfig) {
          const parsed = JSON.parse(savedConfig);
          // Ensure all required fields exist in the config
          setConfig(prev => ({
            ...prev,
            ...parsed,
            qnaPairs: parsed.qnaPairs || prev.qnaPairs,
            features: parsed.features || prev.features
          }));
        }
      } catch (error) {
        console.error('Failed to load chatbot config:', error);
      } finally {
        setIsLoading(false);
      }
    };

    loadConfig();
  }, []);

  // Save configuration
  const handleSave = async () => {
    setIsSaving(true);
    
    try {
      console.log('Saving config to localStorage:', config);
      console.log('Q&A pairs being saved:', config.qnaPairs);
      
      // Save to localStorage for now (could be extended to API)
      localStorage.setItem('chatbotConfig', JSON.stringify(config));
      
      // Verify save
      const saved = localStorage.getItem('chatbotConfig');
      console.log('Verified saved data:', saved);
      
      // Trigger custom event to notify chatbot of changes
      window.dispatchEvent(new CustomEvent('chatbotConfigUpdated'));
      
      toast({
        title: "Chatbot ayarları kaydedildi",
        description: "Değişiklikler başarıyla uygulandı.",
      });
    } catch (error) {
      toast({
        title: "Kaydetme hatası",
        description: "Ayarlar kaydedilemedi. Lütfen tekrar deneyin.",
        variant: "destructive",
      });
    } finally {
      setIsSaving(false);
    }
  };

  // Reset to defaults
  const handleReset = () => {
    setConfig({
      isEnabled: true,
      welcomeMessage: "Merhaba, Jouska Cafe & Restaurant'ya hoş geldiniz! 🌟 Size nasıl yardımcı olabilirim? Menüye göz atmak veya arama yaparak istediğiniz lezzetleri daha hızlı bulmanız için buradayım. Ne arıyorsunuz?",
      autoOpen: true,
      openDelay: 1500,
      quickReplies: [
        "Menüyü görüntüle",
        "Kahvaltılar", 
        "Ana Yemekler",
        "Popüler Ürünler",
        "Vejetaryen",
        "Hızlı Servis"
      ],
      categories: ["kahvaltilar", "ana-yemekler", "icecekler", "tatlilar"],
      searchKeywords: ["popüler", "vejetaryen", "hızlı", "sıcak", "soğuk", "taze"],
      botName: "Jouska Asistan",
      botAvatar: "🤖",
      themeColor: "#f97316",
      qnaPairs: [
        {
          id: "1",
          question: "Açılış saatleriniz nedir?",
          answer: "Haftanın her günü 08:00-23:00 saatleri arasında hizmet veriyoruz.",
          keywords: ["açılış", "saat", "zaman", "kaçta", "kapanış"],
          isActive: true
        },
        {
          id: "2", 
          question: "Teslimat yapıyor musunuz?",
          answer: "Evet, online sipariş ile teslimat hizmeti sunuyoruz. Minimum sipariş tutarı 50₺'dir.",
          keywords: ["teslimat", "sipariş", "online", "getir"],
          isActive: true
        }
      ],
      customerQuestions: [],
      features: {
        orderSystem: {
          enabled: false,
          minOrderAmount: 50,
          deliveryFee: 15,
          estimatedDeliveryTime: "30-45 dakika",
          paymentMethods: ["Nakit", "Kredi Kartı", "Online Ödeme"]
        },
        reservationSystem: {
          enabled: false,
          maxAdvanceDays: 30,
          timeSlots: ["12:00", "12:30", "13:00", "13:30", "19:00", "19:30", "20:00", "20:30"],
          maxPartySize: 8,
          requirePhone: true
        },
        socialFeatures: {
          whatsappEnabled: false,
          whatsappNumber: "",
          instagramEnabled: false,
          instagramHandle: "",
          reviewsEnabled: false
        },
        analytics: {
          enabled: true,
          trackPopularItems: true,
          trackCustomerBehavior: true,
          generateReports: false
        }
      }
    });
  };

  // Add quick reply
  const addQuickReply = () => {
    if (newQuickReply.trim() && !config.quickReplies.includes(newQuickReply.trim())) {
      setConfig(prev => ({
        ...prev,
        quickReplies: [...prev.quickReplies, newQuickReply.trim()]
      }));
      setNewQuickReply("");
    }
  };

  // Remove quick reply
  const removeQuickReply = (reply: string) => {
    setConfig(prev => ({
      ...prev,
      quickReplies: prev.quickReplies.filter(r => r !== reply)
    }));
  };

  // Add search keyword
  const addKeyword = () => {
    if (newKeyword.trim() && !config.searchKeywords.includes(newKeyword.trim())) {
      setConfig(prev => ({
        ...prev,
        searchKeywords: [...prev.searchKeywords, newKeyword.trim()]
      }));
      setNewKeyword("");
    }
  };

  // Remove search keyword
  const removeKeyword = (keyword: string) => {
    setConfig(prev => ({
      ...prev,
      searchKeywords: prev.searchKeywords.filter(k => k !== keyword)
    }));
  };

  // Add Q&A pair
  const addQnAPair = () => {
    if (newQuestion.trim() && newAnswer.trim()) {
      const keywords = newQnaKeywords.split(',').map(k => k.trim()).filter(k => k);
      const newPair: QnAPair = {
        id: Date.now().toString(),
        question: newQuestion.trim(),
        answer: newAnswer.trim(),
        keywords: keywords,
        isActive: true
      };
      
      const updatedConfig = {
        ...config,
        qnaPairs: [...config.qnaPairs, newPair]
      };
      
      setConfig(updatedConfig);
      
      // Otomatik kaydet
      localStorage.setItem('chatbotConfig', JSON.stringify(updatedConfig));
      window.dispatchEvent(new CustomEvent('chatbotConfigUpdated'));
      
      setNewQuestion("");
      setNewAnswer("");
      setNewQnaKeywords("");
      
      toast({
        title: "Soru-Cevap eklendi",
        description: "Yeni Q&A çifti başarıyla eklendi ve kaydedildi.",
      });
    }
  };

  // Update Q&A pair
  const updateQnAPair = () => {
    if (editingQna && newQuestion.trim() && newAnswer.trim()) {
      const keywords = newQnaKeywords.split(',').map(k => k.trim()).filter(k => k);
      
      const updatedConfig = {
        ...config,
        qnaPairs: config.qnaPairs.map(pair => 
          pair.id === editingQna.id 
            ? { ...pair, question: newQuestion.trim(), answer: newAnswer.trim(), keywords }
            : pair
        )
      };
      
      setConfig(updatedConfig);
      
      // Otomatik kaydet
      localStorage.setItem('chatbotConfig', JSON.stringify(updatedConfig));
      window.dispatchEvent(new CustomEvent('chatbotConfigUpdated'));
      
      setEditingQna(null);
      setNewQuestion("");
      setNewAnswer("");
      setNewQnaKeywords("");
      
      toast({
        title: "Soru-Cevap güncellendi",
        description: "Q&A çifti başarıyla güncellendi.",
      });
    }
  };

  // Remove Q&A pair
  const removeQnAPair = (id: string) => {
    const updatedConfig = {
      ...config,
      qnaPairs: config.qnaPairs.filter(pair => pair.id !== id)
    };
    
    setConfig(updatedConfig);
    
    // Otomatik kaydet
    localStorage.setItem('chatbotConfig', JSON.stringify(updatedConfig));
    window.dispatchEvent(new CustomEvent('chatbotConfigUpdated'));
    
    toast({
      title: "Soru-Cevap silindi",
      description: "Q&A çifti başarıyla silindi.",
    });
  };

  // Toggle Q&A pair active status
  const toggleQnAPair = (id: string) => {
    const updatedConfig = {
      ...config,
      qnaPairs: config.qnaPairs.map(pair => 
        pair.id === id ? { ...pair, isActive: !pair.isActive } : pair
      )
    };
    
    setConfig(updatedConfig);
    
    // Otomatik kaydet
    localStorage.setItem('chatbotConfig', JSON.stringify(updatedConfig));
    window.dispatchEvent(new CustomEvent('chatbotConfigUpdated'));
  };

  // Start editing Q&A pair
  const startEditingQna = (pair: QnAPair) => {
    setEditingQna(pair);
    setNewQuestion(pair.question);
    setNewAnswer(pair.answer);
    setNewQnaKeywords(pair.keywords.join(', '));
  };

  // Cancel editing
  const cancelEditing = () => {
    setEditingQna(null);
    setNewQuestion("");
    setNewAnswer("");
    setNewQnaKeywords("");
  };

  if (isLoading) {
    return (
      <Card>
        <CardHeader>
          <div className="flex items-center space-x-2">
            <Loader2 className="h-5 w-5 animate-spin" />
            <CardTitle>Chatbot Ayarları Yükleniyor...</CardTitle>
          </div>
        </CardHeader>
      </Card>
    );
  }

  return (
    <div className="space-y-6">
      {/* Genel Ayarlar */}
      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-2">
              <Bot className="h-5 w-5 text-orange-500" />
              <CardTitle>Genel Chatbot Ayarları</CardTitle>
            </div>
            <div className="flex items-center space-x-2">
              <Button variant="outline" size="sm" onClick={handleReset}>
                <RotateCcw className="h-4 w-4 mr-1" />
                Sıfırla
              </Button>
              <Button size="sm" onClick={handleSave} disabled={isSaving}>
                {isSaving ? (
                  <Loader2 className="h-4 w-4 mr-1 animate-spin" />
                ) : (
                  <Save className="h-4 w-4 mr-1" />
                )}
                Kaydet
              </Button>
            </div>
          </div>
          <CardDescription>
            QR kod tarama sonrası müşterileri karşılayan chatbot ayarlarını yönetin
          </CardDescription>
        </CardHeader>
        
        <CardContent className="space-y-6">
          {/* Aktif/Pasif */}
          <div className="flex items-center justify-between">
            <div className="space-y-0.5">
              <Label htmlFor="chatbot-enabled">Chatbot Aktif</Label>
              <p className="text-sm text-muted-foreground">
                Chatbot sistemini etkinleştir/devre dışı bırak
              </p>
            </div>
            <Switch
              id="chatbot-enabled"
              checked={config.isEnabled}
              onCheckedChange={(checked) => setConfig(prev => ({ ...prev, isEnabled: checked }))}
            />
          </div>

          <Separator />

          {/* Bot İsmi */}
          <div className="space-y-2">
            <Label htmlFor="bot-name">Bot İsmi</Label>
            <Input
              id="bot-name"
              value={config.botName}
              onChange={(e) => setConfig(prev => ({ ...prev, botName: e.target.value }))}
              placeholder="Jouska Asistan"
            />
          </div>

          {/* Hoş Geldiniz Mesajı */}
          <div className="space-y-2">
            <Label htmlFor="welcome-message">Hoş Geldiniz Mesajı</Label>
            <Textarea
              id="welcome-message"
              value={config.welcomeMessage}
              onChange={(e) => setConfig(prev => ({ ...prev, welcomeMessage: e.target.value }))}
              placeholder="Müşterileri karşılayan mesaj..."
              rows={4}
            />
            <p className="text-sm text-muted-foreground">
              Müşteriler QR kod taradığında gördükleri ilk mesaj
            </p>
          </div>

          {/* Otomatik Açılma */}
          <div className="flex items-center justify-between">
            <div className="space-y-0.5">
              <Label htmlFor="auto-open">Otomatik Açılma</Label>
              <p className="text-sm text-muted-foreground">
                QR kod tarandığında chatbot otomatik açılsın
              </p>
            </div>
            <Switch
              id="auto-open"
              checked={config.autoOpen}
              onCheckedChange={(checked) => setConfig(prev => ({ ...prev, autoOpen: checked }))}
            />
          </div>

          {/* Açılma Gecikmesi */}
          {config.autoOpen && (
            <div className="space-y-2">
              <Label htmlFor="open-delay">Açılma Gecikmesi (ms)</Label>
              <Input
                id="open-delay"
                type="number"
                value={config.openDelay}
                onChange={(e) => setConfig(prev => ({ ...prev, openDelay: parseInt(e.target.value) || 1500 }))}
                min="0"
                max="10000"
                step="500"
              />
              <p className="text-sm text-muted-foreground">
                Sayfa yüklendikten sonra chatbot kaç milisaniye bekleyerek açılsın
              </p>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Hızlı Yanıtlar */}
      <Card>
        <CardHeader>
          <div className="flex items-center space-x-2">
            <Zap className="h-5 w-5 text-orange-500" />
            <CardTitle>Hızlı Yanıtlar</CardTitle>
          </div>
          <CardDescription>
            Müşterilere sunulan hazır yanıt seçeneklerini düzenleyin
          </CardDescription>
        </CardHeader>
        
        <CardContent className="space-y-4">
          {/* Yeni Hızlı Yanıt Ekleme */}
          <div className="flex space-x-2">
            <Input
              placeholder="Yeni hızlı yanıt ekle..."
              value={newQuickReply}
              onChange={(e) => setNewQuickReply(e.target.value)}
              onKeyPress={(e) => e.key === 'Enter' && addQuickReply()}
            />
            <Button onClick={addQuickReply} size="sm">
              <Plus className="h-4 w-4" />
            </Button>
          </div>

          {/* Mevcut Hızlı Yanıtlar */}
          <div className="flex flex-wrap gap-2">
            {config.quickReplies.map((reply, index) => (
              <Badge
                key={index}
                variant="secondary"
                className="flex items-center space-x-1 cursor-pointer hover:bg-red-100"
                onClick={() => removeQuickReply(reply)}
              >
                <span>{reply}</span>
                <X className="h-3 w-3" />
              </Badge>
            ))}
          </div>
        </CardContent>
      </Card>

      {/* Arama Anahtar Kelimeleri */}
      <Card>
        <CardHeader>
          <div className="flex items-center space-x-2">
            <Settings className="h-5 w-5 text-orange-500" />
            <CardTitle>Arama Anahtar Kelimeleri</CardTitle>
          </div>
          <CardDescription>
            Chatbot'un önerebileceği arama terimleri
          </CardDescription>
        </CardHeader>
        
        <CardContent className="space-y-4">
          {/* Yeni Anahtar Kelime Ekleme */}
          <div className="flex space-x-2">
            <Input
              placeholder="Yeni anahtar kelime ekle..."
              value={newKeyword}
              onChange={(e) => setNewKeyword(e.target.value)}
              onKeyPress={(e) => e.key === 'Enter' && addKeyword()}
            />
            <Button onClick={addKeyword} size="sm">
              <Plus className="h-4 w-4" />
            </Button>
          </div>

          {/* Mevcut Anahtar Kelimeler */}
          <div className="flex flex-wrap gap-2">
            {config.searchKeywords.map((keyword, index) => (
              <Badge
                key={index}
                variant="outline"
                className="flex items-center space-x-1 cursor-pointer hover:bg-red-100"
                onClick={() => removeKeyword(keyword)}
              >
                <span>{keyword}</span>
                <X className="h-3 w-3" />
              </Badge>
            ))}
          </div>
        </CardContent>
      </Card>

      {/* Soru & Cevap Yönetimi */}
      <Card>
        <CardHeader>
          <div className="flex items-center space-x-2">
            <MessageCircle className="h-5 w-5 text-orange-500" />
            <CardTitle>Soru & Cevap Yönetimi</CardTitle>
          </div>
          <CardDescription>
            Chatbot'un verebileceği özel soru-cevap çiftlerini yönetin
          </CardDescription>
        </CardHeader>
        
        <CardContent className="space-y-6">
          {/* Yeni S&C Ekleme Formu */}
          <div className="space-y-4 p-4 border rounded-lg bg-muted/20">
            <div className="flex items-center justify-between">
              <h4 className="font-medium">
                {editingQna ? 'Soru & Cevap Düzenle' : 'Yeni Soru & Cevap Ekle'}
              </h4>
              {editingQna && (
                <Button variant="outline" size="sm" onClick={cancelEditing}>
                  <X className="h-4 w-4 mr-1" />
                  İptal
                </Button>
              )}
            </div>
            
            <div className="space-y-3">
              <div>
                <Label htmlFor="question">Soru</Label>
                <Input
                  id="question"
                  placeholder="Örn: Açılış saatleriniz nedir?"
                  value={newQuestion}
                  onChange={(e) => setNewQuestion(e.target.value)}
                />
              </div>
              
              <div>
                <Label htmlFor="answer">Cevap</Label>
                <Textarea
                  id="answer"
                  placeholder="Örn: Haftanın her günü 08:00-23:00 saatleri arasında hizmet veriyoruz."
                  value={newAnswer}
                  onChange={(e) => setNewAnswer(e.target.value)}
                  rows={3}
                />
              </div>
              
              <div>
                <Label htmlFor="keywords">Anahtar Kelimeler (virgülle ayırın)</Label>
                <Input
                  id="keywords"
                  placeholder="açılış, saat, zaman, kaçta, kapanış"
                  value={newQnaKeywords}
                  onChange={(e) => setNewQnaKeywords(e.target.value)}
                />
                <p className="text-xs text-muted-foreground mt-1">
                  Bu kelimeler kullanıcı sorularında aranacak ve eşleştirilecek
                </p>
              </div>
              
              <Button 
                onClick={editingQna ? updateQnAPair : addQnAPair}
                disabled={!newQuestion.trim() || !newAnswer.trim()}
                className="w-full"
              >
                <Plus className="h-4 w-4 mr-1" />
                {editingQna ? 'Güncelle' : 'Ekle'}
              </Button>
            </div>
          </div>

          {/* Mevcut S&C Listesi */}
          <div className="space-y-3">
            <h4 className="font-medium">Mevcut Soru & Cevaplar ({config.qnaPairs.length})</h4>
            
            {config.qnaPairs.length === 0 ? (
              <p className="text-muted-foreground text-center py-4">
                Henüz soru-cevap çifti eklenmemiş
              </p>
            ) : (
              <div className="space-y-3">
                {config.qnaPairs.map((pair) => (
                  <div
                    key={pair.id}
                    className={`border rounded-lg p-4 ${
                      pair.isActive ? 'bg-background' : 'bg-muted/50'
                    }`}
                  >
                    <div className="flex items-start justify-between mb-2">
                      <div className="flex items-center space-x-2">
                        <Switch
                          checked={pair.isActive}
                          onCheckedChange={() => toggleQnAPair(pair.id)}
                        />
                        <span className={`text-sm ${pair.isActive ? 'text-foreground' : 'text-muted-foreground'}`}>
                          {pair.isActive ? 'Aktif' : 'Pasif'}
                        </span>
                      </div>
                      <div className="flex space-x-1">
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => startEditingQna(pair)}
                        >
                          Düzenle
                        </Button>
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => removeQnAPair(pair.id)}
                          className="text-red-600 hover:text-red-700 hover:bg-red-50"
                        >
                          <X className="h-4 w-4" />
                        </Button>
                      </div>
                    </div>
                    
                    <div className="space-y-2">
                      <div>
                        <p className="text-sm font-medium text-blue-600">S: {pair.question}</p>
                      </div>
                      <div>
                        <p className="text-sm text-foreground">C: {pair.answer}</p>
                      </div>
                      {pair.keywords.length > 0 && (
                        <div className="flex flex-wrap gap-1 mt-2">
                          {pair.keywords.map((keyword, index) => (
                            <Badge key={index} variant="secondary" className="text-xs">
                              {keyword}
                            </Badge>
                          ))}
                        </div>
                      )}
                    </div>
                  </div>
                ))}
              </div>
            )}
          </div>
        </CardContent>
      </Card>

      {/* Önizleme */}
      <Card>
        <CardHeader>
          <div className="flex items-center space-x-2">
            <MessageCircle className="h-5 w-5 text-orange-500" />
            <CardTitle>Chatbot Önizlemesi</CardTitle>
          </div>
          <CardDescription>
            Mevcut ayarlarla chatbot nasıl görünecek
          </CardDescription>
        </CardHeader>
        
        <CardContent>
          <div className="border rounded-lg p-4 bg-muted/20">
            <div className="flex items-center space-x-2 mb-3">
              <div className="w-8 h-8 bg-orange-500 rounded-full flex items-center justify-center text-white text-sm font-semibold">
                {config.botAvatar}
              </div>
              <div>
                <p className="font-medium text-sm">{config.botName}</p>
                <p className="text-xs text-muted-foreground">Menü Rehberiniz</p>
              </div>
            </div>
            
            <div className="bg-background rounded-lg p-3 mb-3">
              <p className="text-sm">{config.welcomeMessage}</p>
            </div>
            
            <div className="flex flex-wrap gap-1">
              {config.quickReplies.slice(0, 3).map((reply, index) => (
                <Badge key={index} variant="secondary" className="text-xs">
                  {reply}
                </Badge>
              ))}
              {config.quickReplies.length > 3 && (
                <Badge variant="outline" className="text-xs">
                  +{config.quickReplies.length - 3} daha
                </Badge>
              )}
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Müşteri Soruları */}
      <Card>
        <CardHeader>
          <div className="flex items-center space-x-2">
            <MessageSquare className="h-5 w-5 text-blue-500" />
            <CardTitle>Müşteri Soruları İstatistikleri</CardTitle>
          </div>
          <CardDescription>
            Müşterilerin sorduğu soruları görüntüleyin ve analiz edin
          </CardDescription>
        </CardHeader>
        
        <CardContent>
          <div className="space-y-4">
            {config.customerQuestions && config.customerQuestions.length > 0 ? (
              <div className="space-y-3">
                {config.customerQuestions
                  .sort((a, b) => b.frequency - a.frequency)
                  .map((question) => (
                    <div key={question.id} className="border rounded-lg p-4 space-y-2">
                      <div className="flex items-start justify-between">
                        <div className="flex-1">
                          <div className="flex items-center gap-2 mb-1">
                            <span className="font-medium text-sm">{question.question}</span>
                            <Badge variant={question.hasAnswer ? "default" : "destructive"} className="text-xs">
                              {question.hasAnswer ? "Cevaplandı" : "Cevapsız"}
                            </Badge>
                            <Badge variant="outline" className="text-xs">
                              {question.frequency}x soruldu
                            </Badge>
                          </div>
                          <p className="text-xs text-muted-foreground">
                            Son: {new Date(question.timestamp).toLocaleString('tr-TR')}
                          </p>
                          {question.hasAnswer && question.answerGiven && (
                            <div className="mt-2 p-2 bg-muted rounded text-sm">
                              <strong>Verilen Cevap:</strong> {question.answerGiven}
                            </div>
                          )}
                        </div>
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => {
                            setNewQuestion(question.question);
                            setNewAnswer(question.answerGiven || "");
                            setNewQnaKeywords("");
                          }}
                        >
                          <Plus className="h-4 w-4 mr-1" />
                          Q&A'ya Ekle
                        </Button>
                      </div>
                    </div>
                  ))}
              </div>
            ) : (
              <div className="text-center py-8 text-muted-foreground">
                <MessageSquare className="h-12 w-12 mx-auto mb-4 opacity-50" />
                <div>Henüz müşteri sorusu kaydedilmedi.</div>
                <p className="text-sm mt-1">Müşteriler chatbot'ta soru sordukça burada görünecek.</p>
              </div>
            )}
          </div>
        </CardContent>
      </Card>

      {/* Sipariş Sistemi Ayarları */}
      <Card>
        <CardHeader>
          <div className="flex items-center space-x-2">
            <ShoppingCart className="h-5 w-5 text-green-500" />
            <CardTitle>Sipariş Sistemi</CardTitle>
          </div>
          <CardDescription>
            Chatbot üzerinden sipariş alma sistemini yönetin
          </CardDescription>
        </CardHeader>
        
        <CardContent className="space-y-4">
          <div className="flex items-center justify-between">
            <div className="space-y-0.5">
              <Label>Sipariş Sistemi Aktif</Label>
              <p className="text-sm text-muted-foreground">
                Müşterilerin chatbot üzerinden sipariş verebilmesini sağlar
              </p>
            </div>
            <Switch
              checked={config.features.orderSystem.enabled}
              onCheckedChange={(checked) => 
                setConfig(prev => ({
                  ...prev,
                  features: {
                    ...prev.features,
                    orderSystem: { ...prev.features.orderSystem, enabled: checked }
                  }
                }))
              }
            />
          </div>

          {config.features.orderSystem.enabled && (
            <div className="space-y-4 pt-4 border-t">
              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label>Minimum Sipariş Tutarı (₺)</Label>
                  <Input
                    type="number"
                    value={config.features.orderSystem.minOrderAmount}
                    onChange={(e) => 
                      setConfig(prev => ({
                        ...prev,
                        features: {
                          ...prev.features,
                          orderSystem: { ...prev.features.orderSystem, minOrderAmount: Number(e.target.value) }
                        }
                      }))
                    }
                  />
                </div>
                <div className="space-y-2">
                  <Label>Teslimat Ücreti (₺)</Label>
                  <Input
                    type="number"
                    value={config.features.orderSystem.deliveryFee}
                    onChange={(e) => 
                      setConfig(prev => ({
                        ...prev,
                        features: {
                          ...prev.features,
                          orderSystem: { ...prev.features.orderSystem, deliveryFee: Number(e.target.value) }
                        }
                      }))
                    }
                  />
                </div>
              </div>
              
              <div className="space-y-2">
                <Label>Tahmini Teslimat Süresi</Label>
                <Input
                  value={config.features.orderSystem.estimatedDeliveryTime}
                  onChange={(e) => 
                    setConfig(prev => ({
                      ...prev,
                      features: {
                        ...prev.features,
                        orderSystem: { ...prev.features.orderSystem, estimatedDeliveryTime: e.target.value }
                      }
                    }))
                  }
                  placeholder="30-45 dakika"
                />
              </div>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Rezervasyon Sistemi Ayarları */}
      <Card>
        <CardHeader>
          <div className="flex items-center space-x-2">
            <Calendar className="h-5 w-5 text-blue-500" />
            <CardTitle>Rezervasyon Sistemi</CardTitle>
          </div>
          <CardDescription>
            Masa rezervasyon sistemini yönetin
          </CardDescription>
        </CardHeader>
        
        <CardContent className="space-y-4">
          <div className="flex items-center justify-between">
            <div className="space-y-0.5">
              <Label>Rezervasyon Sistemi Aktif</Label>
              <p className="text-sm text-muted-foreground">
                Müşterilerin chatbot üzerinden rezervasyon yapabilmesini sağlar
              </p>
            </div>
            <Switch
              checked={config.features.reservationSystem.enabled}
              onCheckedChange={(checked) => 
                setConfig(prev => ({
                  ...prev,
                  features: {
                    ...prev.features,
                    reservationSystem: { ...prev.features.reservationSystem, enabled: checked }
                  }
                }))
              }
            />
          </div>

          {config.features.reservationSystem.enabled && (
            <div className="space-y-4 pt-4 border-t">
              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label>Maksimum İleri Tarih (Gün)</Label>
                  <Input
                    type="number"
                    value={config.features.reservationSystem.maxAdvanceDays}
                    onChange={(e) => 
                      setConfig(prev => ({
                        ...prev,
                        features: {
                          ...prev.features,
                          reservationSystem: { ...prev.features.reservationSystem, maxAdvanceDays: Number(e.target.value) }
                        }
                      }))
                    }
                  />
                </div>
                <div className="space-y-2">
                  <Label>Maksimum Kişi Sayısı</Label>
                  <Input
                    type="number"
                    value={config.features.reservationSystem.maxPartySize}
                    onChange={(e) => 
                      setConfig(prev => ({
                        ...prev,
                        features: {
                          ...prev.features,
                          reservationSystem: { ...prev.features.reservationSystem, maxPartySize: Number(e.target.value) }
                        }
                      }))
                    }
                  />
                </div>
              </div>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Sosyal Medya Ayarları */}
      <Card>
        <CardHeader>
          <div className="flex items-center space-x-2">
            <Share2 className="h-5 w-5 text-purple-500" />
            <CardTitle>Sosyal Medya Entegrasyonu</CardTitle>
          </div>
          <CardDescription>
            WhatsApp ve Instagram entegrasyonlarını yönetin
          </CardDescription>
        </CardHeader>
        
        <CardContent className="space-y-4">
          <div className="flex items-center justify-between">
            <div className="space-y-0.5">
              <Label>WhatsApp Sipariş Hattı</Label>
              <p className="text-sm text-muted-foreground">
                WhatsApp üzerinden sipariş alma sistemini aktifleştir
              </p>
            </div>
            <Switch
              checked={config.features.socialFeatures.whatsappEnabled}
              onCheckedChange={(checked) => 
                setConfig(prev => ({
                  ...prev,
                  features: {
                    ...prev.features,
                    socialFeatures: { ...prev.features.socialFeatures, whatsappEnabled: checked }
                  }
                }))
              }
            />
          </div>

          {config.features.socialFeatures.whatsappEnabled && (
            <div className="space-y-2">
              <Label>WhatsApp Telefon Numarası</Label>
              <Input
                value={config.features.socialFeatures.whatsappNumber}
                onChange={(e) => 
                  setConfig(prev => ({
                    ...prev,
                    features: {
                      ...prev.features,
                      socialFeatures: { ...prev.features.socialFeatures, whatsappNumber: e.target.value }
                    }
                  }))
                }
                placeholder="+90 555 123 45 67"
              />
            </div>
          )}

          <div className="flex items-center justify-between">
            <div className="space-y-0.5">
              <Label>Instagram Entegrasyonu</Label>
              <p className="text-sm text-muted-foreground">
                Instagram hesabınızı menüde gösterin
              </p>
            </div>
            <Switch
              checked={config.features.socialFeatures.instagramEnabled}
              onCheckedChange={(checked) => 
                setConfig(prev => ({
                  ...prev,
                  features: {
                    ...prev.features,
                    socialFeatures: { ...prev.features.socialFeatures, instagramEnabled: checked }
                  }
                }))
              }
            />
          </div>

          {config.features.socialFeatures.instagramEnabled && (
            <div className="space-y-2">
              <Label>Instagram Kullanıcı Adı</Label>
              <Input
                value={config.features.socialFeatures.instagramHandle}
                onChange={(e) => 
                  setConfig(prev => ({
                    ...prev,
                    features: {
                      ...prev.features,
                      socialFeatures: { ...prev.features.socialFeatures, instagramHandle: e.target.value }
                    }
                  }))
                }
                placeholder="@jouska_restaurant"
              />
            </div>
          )}
        </CardContent>
      </Card>

      {/* Analytics Ayarları */}
      <Card>
        <CardHeader>
          <div className="flex items-center space-x-2">
            <BarChart3 className="h-5 w-5 text-indigo-500" />
            <CardTitle>Analytics ve Raporlama</CardTitle>
          </div>
          <CardDescription>
            Müşteri davranışı ve popüler ürün takibini yönetin
          </CardDescription>
        </CardHeader>
        
        <CardContent className="space-y-4">
          <div className="flex items-center justify-between">
            <div className="space-y-0.5">
              <Label>Popüler Ürün Takibi</Label>
              <p className="text-sm text-muted-foreground">
                En çok görüntülenen ürünleri takip et
              </p>
            </div>
            <Switch
              checked={config.features.analytics.trackPopularItems}
              onCheckedChange={(checked) => 
                setConfig(prev => ({
                  ...prev,
                  features: {
                    ...prev.features,
                    analytics: { ...prev.features.analytics, trackPopularItems: checked }
                  }
                }))
              }
            />
          </div>

          <div className="flex items-center justify-between">
            <div className="space-y-0.5">
              <Label>Müşteri Davranış Analizi</Label>
              <p className="text-sm text-muted-foreground">
                Chatbot etkileşimlerini ve kategori tercihlerini analiz et
              </p>
            </div>
            <Switch
              checked={config.features.analytics.trackCustomerBehavior}
              onCheckedChange={(checked) => 
                setConfig(prev => ({
                  ...prev,
                  features: {
                    ...prev.features,
                    analytics: { ...prev.features.analytics, trackCustomerBehavior: checked }
                  }
                }))
              }
            />
          </div>

          <div className="flex items-center justify-between">
            <div className="space-y-0.5">
              <Label>Otomatik Raporlama</Label>
              <p className="text-sm text-muted-foreground">
                Haftalık/aylık otomatik raporlar oluştur
              </p>
            </div>
            <Switch
              checked={config.features.analytics.generateReports}
              onCheckedChange={(checked) => 
                setConfig(prev => ({
                  ...prev,
                  features: {
                    ...prev.features,
                    analytics: { ...prev.features.analytics, generateReports: checked }
                  }
                }))
              }
            />
          </div>
        </CardContent>
      </Card>
    </div>
  );
}

export default ChatbotSettings;