import React, { useState } from "react";
import { Category } from "@shared/schema";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Dialog, DialogContent, DialogFooter, DialogHeader, DialogTitle, DialogClose } from "@/components/ui/dialog";
import { Edit, Plus, Trash2, X } from "lucide-react";
import { useQueryClient } from "@tanstack/react-query";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";

interface CategoryManagementProps {
  categories: Category[];
}

export const CategoryManagement = ({ categories }: CategoryManagementProps) => {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false);
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false);
  const [isDeleteDialogOpen, setIsDeleteDialogOpen] = useState(false);
  const [selectedCategory, setSelectedCategory] = useState<Category | null>(null);
  const [formData, setFormData] = useState({
    name: "",
    slug: "",
    order: 0
  });

  // Open add dialog
  const handleOpenAddDialog = () => {
    setFormData({ name: "", slug: "", order: categories.length + 1 });
    setIsAddDialogOpen(true);
  };

  // Open edit dialog
  const handleOpenEditDialog = (category: Category) => {
    setSelectedCategory(category);
    setFormData({
      name: category.name,
      slug: category.slug,
      order: category.order
    });
    setIsEditDialogOpen(true);
  };

  // Open delete dialog
  const handleOpenDeleteDialog = (category: Category) => {
    setSelectedCategory(category);
    setIsDeleteDialogOpen(true);
  };

  // Update form data
  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: name === "order" ? parseInt(value) || 0 : value
    }));
  };

  // Generate slug from name
  const generateSlug = () => {
    const slug = formData.name
      .toLowerCase()
      .replace(/[^\w\s-]/g, '')
      .replace(/[\s_-]+/g, '-')
      .replace(/^-+|-+$/g, '');
    
    setFormData(prev => ({ ...prev, slug }));
  };

  // Add new category
  const handleAddCategory = async () => {
    try {
      await apiRequest("/api/categories", {
        method: "POST",
        body: JSON.stringify(formData)
      });

      queryClient.invalidateQueries({ queryKey: ['/api/categories'] });
      toast({
        title: "Kategori eklendi",
        description: `${formData.name} kategorisi başarıyla eklendi.`
      });
      setIsAddDialogOpen(false);
    } catch (error) {
      toast({
        title: "Hata",
        description: "Kategori eklenirken bir hata oluştu.",
        variant: "destructive"
      });
    }
  };

  // Update category
  const handleUpdateCategory = async () => {
    if (!selectedCategory) return;
    
    try {
      await apiRequest(`/api/categories/${selectedCategory.id}`, {
        method: "PATCH",
        body: JSON.stringify(formData)
      });

      queryClient.invalidateQueries({ queryKey: ['/api/categories'] });
      toast({
        title: "Kategori güncellendi",
        description: `${formData.name} kategorisi başarıyla güncellendi.`
      });
      setIsEditDialogOpen(false);
    } catch (error) {
      toast({
        title: "Hata",
        description: "Kategori güncellenirken bir hata oluştu.",
        variant: "destructive"
      });
    }
  };

  // Delete category
  const handleDeleteCategory = async () => {
    if (!selectedCategory) return;
    
    try {
      await apiRequest(`/api/categories/${selectedCategory.id}`, {
        method: "DELETE"
      });

      queryClient.invalidateQueries({ queryKey: ['/api/categories'] });
      toast({
        title: "Kategori silindi",
        description: `${selectedCategory.name} kategorisi ve bağlı tüm menü öğeleri silindi.`
      });
      setIsDeleteDialogOpen(false);
    } catch (error) {
      toast({
        title: "Hata",
        description: "Kategori silinirken bir hata oluştu.",
        variant: "destructive"
      });
    }
  };

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <h2 className="text-2xl font-bold text-primary font-display">Kategoriler</h2>
        <Button onClick={handleOpenAddDialog} className="flex items-center gap-1">
          <Plus className="h-4 w-4" /> Kategori Ekle
        </Button>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Tüm Kategoriler</CardTitle>
          <CardDescription>
            Menünüzün kategorilerini yönetin. Kategoriler silindiğinde, içlerindeki tüm menü öğeleri de silinir.
          </CardDescription>
        </CardHeader>
        <CardContent>
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>Sıra</TableHead>
                <TableHead>Kategori Adı</TableHead>
                <TableHead>Slug</TableHead>
                <TableHead className="text-right">İşlemler</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {categories.length === 0 ? (
                <TableRow>
                  <TableCell colSpan={4} className="text-center py-4 text-gray-500">
                    Henüz kategori bulunmuyor
                  </TableCell>
                </TableRow>
              ) : (
                categories
                  .sort((a, b) => a.order - b.order)
                  .map((category) => (
                    <TableRow key={category.id}>
                      <TableCell>{category.order}</TableCell>
                      <TableCell className="font-medium">{category.name}</TableCell>
                      <TableCell>{category.slug}</TableCell>
                      <TableCell className="text-right">
                        <div className="flex justify-end gap-2">
                          <Button 
                            variant="ghost" 
                            size="sm"
                            onClick={() => handleOpenEditDialog(category)}
                          >
                            <Edit className="h-4 w-4" />
                          </Button>
                          <Button 
                            variant="ghost" 
                            size="sm" 
                            className="text-red-500 hover:text-red-700 hover:bg-red-50"
                            onClick={() => handleOpenDeleteDialog(category)}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
              )}
            </TableBody>
          </Table>
        </CardContent>
      </Card>

      {/* Add Category Dialog */}
      <Dialog open={isAddDialogOpen} onOpenChange={setIsAddDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Yeni Kategori Ekle</DialogTitle>
            <DialogClose className="absolute right-4 top-4">
              <X className="h-4 w-4" />
            </DialogClose>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <label htmlFor="name">Kategori Adı</label>
              <Input
                id="name"
                name="name"
                value={formData.name}
                onChange={handleInputChange}
                onBlur={generateSlug}
              />
            </div>
            <div className="space-y-2">
              <label htmlFor="slug">Slug</label>
              <Input
                id="slug"
                name="slug"
                value={formData.slug}
                onChange={handleInputChange}
              />
              <p className="text-xs text-gray-500">
                URL'de kullanılacak özel isim. Boş bırakırsanız kategori adından otomatik oluşturulur.
              </p>
            </div>
            <div className="space-y-2">
              <label htmlFor="order">Sıralama</label>
              <Input
                id="order"
                name="order"
                type="number"
                value={formData.order}
                onChange={handleInputChange}
              />
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setIsAddDialogOpen(false)}>
              İptal
            </Button>
            <Button onClick={handleAddCategory}>
              Kategori Ekle
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Edit Category Dialog */}
      <Dialog open={isEditDialogOpen} onOpenChange={setIsEditDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Kategori Düzenle</DialogTitle>
            <DialogClose className="absolute right-4 top-4">
              <X className="h-4 w-4" />
            </DialogClose>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <label htmlFor="edit-name">Kategori Adı</label>
              <Input
                id="edit-name"
                name="name"
                value={formData.name}
                onChange={handleInputChange}
                onBlur={generateSlug}
              />
            </div>
            <div className="space-y-2">
              <label htmlFor="edit-slug">Slug</label>
              <Input
                id="edit-slug"
                name="slug"
                value={formData.slug}
                onChange={handleInputChange}
              />
            </div>
            <div className="space-y-2">
              <label htmlFor="edit-order">Sıralama</label>
              <Input
                id="edit-order"
                name="order"
                type="number"
                value={formData.order}
                onChange={handleInputChange}
              />
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setIsEditDialogOpen(false)}>
              İptal
            </Button>
            <Button onClick={handleUpdateCategory}>
              Güncelle
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Delete Category Dialog */}
      <Dialog open={isDeleteDialogOpen} onOpenChange={setIsDeleteDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Kategori Sil</DialogTitle>
            <DialogClose className="absolute right-4 top-4">
              <X className="h-4 w-4" />
            </DialogClose>
          </DialogHeader>
          <div className="py-4">
            <p className="mb-2">
              <strong>{selectedCategory?.name}</strong> kategorisini silmek istediğinizden emin misiniz?
            </p>
            <p className="text-red-500">
              Bu işlem geri alınamaz ve kategoriye ait tüm menü öğeleri de silinecektir.
            </p>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setIsDeleteDialogOpen(false)}>
              İptal
            </Button>
            <Button variant="destructive" onClick={handleDeleteCategory}>
              Sil
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
};