import React, { useState } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useQuery } from "@tanstack/react-query";
import { 
  BarChart3, 
  TrendingUp, 
  TrendingDown,
  Users, 
  ShoppingCart, 
  DollarSign,
  Calendar,
  Eye,
  Clock,
  Star,
  Target,
  Package
} from "lucide-react";

interface AnalyticsData {
  totalOrders: number;
  totalRevenue: string;
  totalCustomers: number;
  averageOrderValue: string;
  popularItems: Array<{
    name: string;
    orders: number;
    revenue: string;
  }>;
  recentOrders: Array<{
    id: number;
    orderNumber: string;
    customerName: string;
    totalAmount: string;
    status: string;
    createdAt: string;
  }>;
  monthlyStats: Array<{
    month: string;
    orders: number;
    revenue: number;
  }>;
  customerStats: {
    newCustomers: number;
    returningCustomers: number;
  };
}

export default function AnalyticsDashboard() {
  const [dateRange, setDateRange] = useState("7d");
  
  // Fetch analytics data
  const { data: analytics, isLoading } = useQuery<AnalyticsData>({
    queryKey: ["/api/analytics", dateRange],
  });

  if (isLoading) {
    return (
      <div className="space-y-6">
        <div className="animate-pulse">
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
            {[...Array(4)].map((_, i) => (
              <div key={i} className="bg-gray-200 h-32 rounded-lg"></div>
            ))}
          </div>
        </div>
      </div>
    );
  }

  const mockAnalytics: AnalyticsData = {
    totalOrders: 156,
    totalRevenue: "12,450.75",
    totalCustomers: 89,
    averageOrderValue: "79.85",
    popularItems: [
      { name: "Türk Kahvesi", orders: 45, revenue: "675.00" },
      { name: "Menemen", orders: 38, revenue: "950.00" },
      { name: "Künefe", orders: 32, revenue: "800.00" },
      { name: "Çay", orders: 78, revenue: "390.00" },
      { name: "Baklava", orders: 25, revenue: "750.00" },
    ],
    recentOrders: [
      { id: 1, orderNumber: "ORD-001", customerName: "Ahmet Yılmaz", totalAmount: "145.50", status: "completed", createdAt: "2024-01-15T10:30:00Z" },
      { id: 2, orderNumber: "ORD-002", customerName: "Fatma Demir", totalAmount: "89.75", status: "pending", createdAt: "2024-01-15T11:15:00Z" },
      { id: 3, orderNumber: "ORD-003", customerName: "Mehmet Öz", totalAmount: "67.25", status: "completed", createdAt: "2024-01-15T12:00:00Z" },
    ],
    monthlyStats: [
      { month: "Oca", orders: 120, revenue: 8500 },
      { month: "Şub", orders: 145, revenue: 9200 },
      { month: "Mar", orders: 156, revenue: 12450 },
    ],
    customerStats: {
      newCustomers: 23,
      returningCustomers: 66,
    }
  };

  const data = analytics || mockAnalytics;

  const getStatusColor = (status: string) => {
    switch (status) {
      case "completed": return "default";
      case "pending": return "secondary";
      case "cancelled": return "destructive";
      default: return "secondary";
    }
  };

  const getStatusText = (status: string) => {
    switch (status) {
      case "completed": return "Tamamlandı";
      case "pending": return "Bekliyor";
      case "cancelled": return "İptal";
      default: return status;
    }
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex justify-between items-center">
        <div>
          <h2 className="text-2xl font-bold tracking-tight">Analytics Dashboard</h2>
          <p className="text-muted-foreground">
            İşletme performansınızı izleyin ve analiz edin
          </p>
        </div>
        <Select value={dateRange} onValueChange={setDateRange}>
          <SelectTrigger className="w-40">
            <SelectValue />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="7d">Son 7 Gün</SelectItem>
            <SelectItem value="30d">Son 30 Gün</SelectItem>
            <SelectItem value="90d">Son 3 Ay</SelectItem>
            <SelectItem value="1y">Son Yıl</SelectItem>
          </SelectContent>
        </Select>
      </div>

      {/* Key Metrics */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Toplam Sipariş</p>
                <p className="text-2xl font-bold">{data.totalOrders}</p>
                <div className="flex items-center text-xs text-green-600">
                  <TrendingUp className="h-3 w-3 mr-1" />
                  +12% bu ay
                </div>
              </div>
              <ShoppingCart className="h-8 w-8 text-blue-600" />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Toplam Ciro</p>
                <p className="text-2xl font-bold">{data.totalRevenue} TL</p>
                <div className="flex items-center text-xs text-green-600">
                  <TrendingUp className="h-3 w-3 mr-1" />
                  +18% bu ay
                </div>
              </div>
              <DollarSign className="h-8 w-8 text-green-600" />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Toplam Müşteri</p>
                <p className="text-2xl font-bold">{data.totalCustomers}</p>
                <div className="flex items-center text-xs text-green-600">
                  <TrendingUp className="h-3 w-3 mr-1" />
                  +8% bu ay
                </div>
              </div>
              <Users className="h-8 w-8 text-purple-600" />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Ortalama Sipariş</p>
                <p className="text-2xl font-bold">{data.averageOrderValue} TL</p>
                <div className="flex items-center text-xs text-red-600">
                  <TrendingDown className="h-3 w-3 mr-1" />
                  -3% bu ay
                </div>
              </div>
              <Target className="h-8 w-8 text-orange-600" />
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Charts and Analytics */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Popular Items */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center">
              <Package className="h-5 w-5 mr-2" />
              Popüler Ürünler
            </CardTitle>
            <CardDescription>En çok sipariş edilen ürünler</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {data.popularItems.map((item, index) => (
                <div key={index} className="flex items-center justify-between">
                  <div className="flex items-center space-x-3">
                    <div className="w-8 h-8 bg-primary/10 rounded-full flex items-center justify-center text-sm font-medium">
                      {index + 1}
                    </div>
                    <div>
                      <p className="font-medium">{item.name}</p>
                      <p className="text-sm text-muted-foreground">{item.orders} sipariş</p>
                    </div>
                  </div>
                  <div className="text-right">
                    <p className="font-medium">{item.revenue} TL</p>
                  </div>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>

        {/* Customer Distribution */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center">
              <Users className="h-5 w-5 mr-2" />
              Müşteri Dağılımı
            </CardTitle>
            <CardDescription>Yeni ve dönen müşteri oranları</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <div className="flex items-center justify-between">
                <div className="flex items-center space-x-3">
                  <div className="w-4 h-4 bg-blue-600 rounded-full"></div>
                  <span>Yeni Müşteriler</span>
                </div>
                <div className="text-right">
                  <p className="font-bold">{data.customerStats.newCustomers}</p>
                  <p className="text-sm text-muted-foreground">
                    %{Math.round((data.customerStats.newCustomers / data.totalCustomers) * 100)}
                  </p>
                </div>
              </div>
              <div className="flex items-center justify-between">
                <div className="flex items-center space-x-3">
                  <div className="w-4 h-4 bg-green-600 rounded-full"></div>
                  <span>Dönen Müşteriler</span>
                </div>
                <div className="text-right">
                  <p className="font-bold">{data.customerStats.returningCustomers}</p>
                  <p className="text-sm text-muted-foreground">
                    %{Math.round((data.customerStats.returningCustomers / data.totalCustomers) * 100)}
                  </p>
                </div>
              </div>
              
              {/* Progress bars */}
              <div className="space-y-2 pt-2">
                <div className="w-full bg-gray-200 rounded-full h-2">
                  <div 
                    className="bg-blue-600 h-2 rounded-full" 
                    style={{ width: `${(data.customerStats.newCustomers / data.totalCustomers) * 100}%` }}
                  ></div>
                </div>
                <div className="w-full bg-gray-200 rounded-full h-2">
                  <div 
                    className="bg-green-600 h-2 rounded-full" 
                    style={{ width: `${(data.customerStats.returningCustomers / data.totalCustomers) * 100}%` }}
                  ></div>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Recent Orders */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center">
            <Clock className="h-5 w-5 mr-2" />
            Son Siparişler
          </CardTitle>
          <CardDescription>En son alınan siparişler</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            {data.recentOrders.map((order) => (
              <div key={order.id} className="flex items-center justify-between p-3 border rounded-lg">
                <div className="flex items-center space-x-4">
                  <div>
                    <p className="font-medium">{order.orderNumber}</p>
                    <p className="text-sm text-muted-foreground">{order.customerName}</p>
                  </div>
                </div>
                <div className="flex items-center space-x-4">
                  <Badge variant={getStatusColor(order.status)}>
                    {getStatusText(order.status)}
                  </Badge>
                  <div className="text-right">
                    <p className="font-medium">{order.totalAmount} TL</p>
                    <p className="text-sm text-muted-foreground">
                      {new Date(order.createdAt).toLocaleDateString('tr-TR')}
                    </p>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>

      {/* Monthly Trends */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center">
            <BarChart3 className="h-5 w-5 mr-2" />
            Aylık Trendler
          </CardTitle>
          <CardDescription>Son 3 ayın performans verileri</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            {data.monthlyStats.map((month) => (
              <div key={month.month} className="flex items-center justify-between">
                <div className="flex items-center space-x-3">
                  <div className="w-12 text-center">
                    <span className="font-medium">{month.month}</span>
                  </div>
                  <div className="flex-1">
                    <div className="w-full bg-gray-200 rounded-full h-2">
                      <div 
                        className="bg-primary h-2 rounded-full" 
                        style={{ width: `${(month.orders / Math.max(...data.monthlyStats.map(m => m.orders))) * 100}%` }}
                      ></div>
                    </div>
                  </div>
                </div>
                <div className="text-right min-w-0 flex-shrink-0 ml-4">
                  <p className="font-medium">{month.orders} sipariş</p>
                  <p className="text-sm text-muted-foreground">{month.revenue.toLocaleString()} TL</p>
                </div>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>

      {/* Performance Insights */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center">
            <Star className="h-5 w-5 mr-2" />
            Performans İçgörüleri
          </CardTitle>
          <CardDescription>İşletmeniz için öneriler ve analizler</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="p-4 bg-green-50 border border-green-200 rounded-lg">
              <div className="flex items-center space-x-2 mb-2">
                <TrendingUp className="h-4 w-4 text-green-600" />
                <span className="text-sm font-medium text-green-800">Güçlü Taraf</span>
              </div>
              <p className="text-sm text-green-700">
                Müşteri memnuniyetiniz yüksek ve sipariş sayınız istikrarlı bir şekilde artıyor.
              </p>
            </div>
            
            <div className="p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
              <div className="flex items-center space-x-2 mb-2">
                <Target className="h-4 w-4 text-yellow-600" />
                <span className="text-sm font-medium text-yellow-800">İyileştirme Alanı</span>
              </div>
              <p className="text-sm text-yellow-700">
                Ortalama sipariş değerini artırmak için combo menüler veya ek ürün önerileri deneyebilirsiniz.
              </p>
            </div>
            
            <div className="p-4 bg-blue-50 border border-blue-200 rounded-lg">
              <div className="flex items-center space-x-2 mb-2">
                <Eye className="h-4 w-4 text-blue-600" />
                <span className="text-sm font-medium text-blue-800">Fırsat</span>
              </div>
              <p className="text-sm text-blue-700">
                Popüler ürünlerinizi öne çıkararak satışları daha da artırabilirsiniz.
              </p>
            </div>
            
            <div className="p-4 bg-purple-50 border border-purple-200 rounded-lg">
              <div className="flex items-center space-x-2 mb-2">
                <Calendar className="h-4 w-4 text-purple-600" />
                <span className="text-sm font-medium text-purple-800">Trend</span>
              </div>
              <p className="text-sm text-purple-700">
                Hafta sonu siparişleriniz %25 daha yüksek. Bu dönemlere özel kampanyalar düşünün.
              </p>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}