import { Switch, Route, useLocation } from "wouter";
import { queryClient } from "./lib/queryClient";
import { QueryClientProvider } from "@tanstack/react-query";
import { Toaster } from "@/components/ui/toaster";
import { TooltipProvider } from "@/components/ui/tooltip";
import { ThemeProvider } from "./components/theme-provider";
import NotFound from "@/pages/not-found";
import Home from "@/pages/home";
import AdminPage from "@/pages/admin";
import AuthPage from "@/pages/auth-page";
import CategoryPage from "@/pages/category-page";
import { AuthProvider } from "@/hooks/use-auth";
import { CartProvider } from "@/hooks/use-cart";
import { ProtectedRoute } from "@/lib/protected-route";
import { WelcomeChatbot } from "@/components/ui/welcome-chatbot";
import { CartButton } from "@/components/cart/cart-button";
import { useState, useEffect } from "react";
import { useQuery } from "@tanstack/react-query";
import { Restaurant } from "@shared/schema";
import { Button } from "@/components/ui/button";
import { MessageCircle } from "lucide-react";

function Router() {
  const [location, setLocation] = useLocation();
  const [isChatbotOpen, setIsChatbotOpen] = useState(false);
  const [searchQuery, setSearchQuery] = useState("");
  
  // Restaurant bilgilerini al
  const restaurantQuery = useQuery<Restaurant>({ 
    queryKey: ['/api/restaurant'] 
  });

  // Chatbot'u otomatik aç
  useEffect(() => {
    const urlParams = new URLSearchParams(window.location.search);
    const isQrScan = urlParams.get('qr') === 'true';
    
    // QR tarama durumunda veya normal sayfa yüklendiğinde chatbot ayarlarını kontrol et
    const checkAutoOpen = () => {
      try {
        const savedConfig = localStorage.getItem('chatbotConfig');
        if (savedConfig) {
          const config = JSON.parse(savedConfig);
          // Chatbot aktif ve otomatik açılma ayarı açık ise
          if (config.isEnabled && config.autoOpen) {
            const delay = config.openDelay || 1500;
            setTimeout(() => {
              setIsChatbotOpen(true);
            }, delay);
          }
        } else {
          // Ayar yoksa varsayılan olarak chatbot'u aç (ilk ziyaret)
          setTimeout(() => {
            setIsChatbotOpen(true);
          }, 1500);
        }
      } catch (error) {
        // Hata durumunda QR tarama varsa varsayılan davranış
        if (isQrScan && location === '/') {
          setTimeout(() => {
            setIsChatbotOpen(true);
          }, 1500);
        }
      }
    };

    if (location === '/') {
      checkAutoOpen();
    }
  }, [location]);

  const handleCategorySelect = (categorySlug: string) => {
    setLocation(`/category/${categorySlug}`);
  };

  const handleSearch = (query: string) => {
    setSearchQuery(query);
    setLocation('/');
  };
  
  return (
    <>
      <Switch>
        <Route path="/" component={Home} />
        <Route path="/auth" component={AuthPage} />
        <Route path="/category/:slug" component={CategoryPage} />
        <ProtectedRoute path="/yonetim-paneli" component={AdminPage} />
        <Route component={NotFound} />
      </Switch>
      
      {/* Global Chatbot - Admin sayfası dışında tüm sayfalarda görünür */}
      {!location.includes('/yonetim-paneli') && (
        <>
          <WelcomeChatbot
            isOpen={isChatbotOpen}
            onClose={() => setIsChatbotOpen(false)}
            onSearch={handleSearch}
            onCategorySelect={handleCategorySelect}
            restaurantName={restaurantQuery.data?.name}
          />
          
          {/* Shopping Cart Button */}
          <CartButton />
          
          {/* Floating Chatbot Button */}
          {!isChatbotOpen && (
            <Button
              className="fixed bottom-6 left-6 sm:bottom-8 sm:left-8 w-14 h-14 sm:w-16 sm:h-16 rounded-full shadow-lg bg-orange-500 hover:bg-orange-600 z-40"
              onClick={() => setIsChatbotOpen(true)}
            >
              <MessageCircle className="w-6 h-6 sm:w-7 sm:h-7 text-white" />
            </Button>
          )}
        </>
      )}
    </>
  );
}

function App() {
  return (
    <QueryClientProvider client={queryClient}>
      <AuthProvider>
        <CartProvider>
          <ThemeProvider defaultTheme="light" storageKey="jouska-theme">
            <TooltipProvider>
              <Toaster />
              <Router />
            </TooltipProvider>
          </ThemeProvider>
        </CartProvider>
      </AuthProvider>
    </QueryClientProvider>
  );
}

export default App;
